/*
 * Decompiled with CFR 0.152.
 */
package com.launchdarkly.client;

import com.google.gson.JsonElement;
import com.google.gson.JsonPrimitive;
import com.launchdarkly.client.Components;
import com.launchdarkly.client.EvaluationDetail;
import com.launchdarkly.client.EvaluationException;
import com.launchdarkly.client.EvaluationReason;
import com.launchdarkly.client.Event;
import com.launchdarkly.client.EventFactory;
import com.launchdarkly.client.EventProcessor;
import com.launchdarkly.client.EventProcessorFactory;
import com.launchdarkly.client.FeatureFlag;
import com.launchdarkly.client.FeatureFlagsState;
import com.launchdarkly.client.FeatureStore;
import com.launchdarkly.client.FeatureStoreClientWrapper;
import com.launchdarkly.client.FeatureStoreFactory;
import com.launchdarkly.client.FlagsStateOption;
import com.launchdarkly.client.LDClientInterface;
import com.launchdarkly.client.LDConfig;
import com.launchdarkly.client.LDUser;
import com.launchdarkly.client.NewRelicReflector;
import com.launchdarkly.client.UpdateProcessor;
import com.launchdarkly.client.UpdateProcessorFactory;
import com.launchdarkly.client.VariationType;
import com.launchdarkly.client.VersionedDataKind;
import com.launchdarkly.shaded.com.google.common.base.Preconditions;
import com.launchdarkly.shaded.org.apache.commons.codec.binary.Hex;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.jar.Attributes;
import java.util.jar.Manifest;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LDClient
implements LDClientInterface {
    private static final Logger logger = LoggerFactory.getLogger(LDClient.class);
    private static final String HMAC_ALGORITHM = "HmacSHA256";
    static final String CLIENT_VERSION = LDClient.getClientVersion();
    private final LDConfig config;
    private final String sdkKey;
    final EventProcessor eventProcessor;
    final UpdateProcessor updateProcessor;
    final FeatureStore featureStore;
    final boolean shouldCloseFeatureStore;

    public LDClient(String sdkKey) {
        this(sdkKey, LDConfig.DEFAULT);
    }

    public LDClient(String sdkKey, LDConfig config) {
        FeatureStore store;
        this.config = Preconditions.checkNotNull(config, "config must not be null");
        this.sdkKey = Preconditions.checkNotNull(sdkKey, "sdkKey must not be null");
        if (config.deprecatedFeatureStore != null) {
            store = config.deprecatedFeatureStore;
            this.shouldCloseFeatureStore = false;
        } else {
            FeatureStoreFactory factory = config.featureStoreFactory == null ? Components.inMemoryFeatureStore() : config.featureStoreFactory;
            store = factory.createFeatureStore();
            this.shouldCloseFeatureStore = true;
        }
        this.featureStore = new FeatureStoreClientWrapper(store);
        EventProcessorFactory epFactory = config.eventProcessorFactory == null ? Components.defaultEventProcessor() : config.eventProcessorFactory;
        this.eventProcessor = epFactory.createEventProcessor(sdkKey, config);
        UpdateProcessorFactory upFactory = config.updateProcessorFactory == null ? Components.defaultUpdateProcessor() : config.updateProcessorFactory;
        this.updateProcessor = upFactory.createUpdateProcessor(sdkKey, config, this.featureStore);
        Future<Void> startFuture = this.updateProcessor.start();
        if (config.startWaitMillis > 0L) {
            if (!config.offline && !config.useLdd) {
                logger.info("Waiting up to " + config.startWaitMillis + " milliseconds for LaunchDarkly client to start...");
            }
            try {
                startFuture.get(config.startWaitMillis, TimeUnit.MILLISECONDS);
            }
            catch (TimeoutException e) {
                logger.error("Timeout encountered waiting for LaunchDarkly client initialization");
            }
            catch (Exception e) {
                logger.error("Exception encountered waiting for LaunchDarkly client initialization: {}", (Object)e.toString());
                logger.debug(e.toString(), (Throwable)e);
            }
            if (!this.updateProcessor.initialized()) {
                logger.warn("LaunchDarkly client was not successfully initialized");
            }
        }
    }

    @Override
    public boolean initialized() {
        return this.updateProcessor.initialized();
    }

    @Override
    public void track(String eventName, LDUser user) {
        this.track(eventName, user, null);
    }

    @Override
    public void track(String eventName, LDUser user, JsonElement data) {
        if (user == null || user.getKey() == null) {
            logger.warn("Track called with null user or null user key!");
        } else {
            this.eventProcessor.sendEvent(EventFactory.DEFAULT.newCustomEvent(eventName, user, data, null));
        }
    }

    @Override
    public void track(String eventName, LDUser user, JsonElement data, double metricValue) {
        if (user == null || user.getKey() == null) {
            logger.warn("Track called with null user or null user key!");
        } else {
            this.eventProcessor.sendEvent(EventFactory.DEFAULT.newCustomEvent(eventName, user, data, metricValue));
        }
    }

    @Override
    public void identify(LDUser user) {
        if (user == null || user.getKey() == null) {
            logger.warn("Identify called with null user or null user key!");
        } else {
            this.eventProcessor.sendEvent(EventFactory.DEFAULT.newIdentifyEvent(user));
        }
    }

    private void sendFlagRequestEvent(Event.FeatureRequest event) {
        this.eventProcessor.sendEvent(event);
        NewRelicReflector.annotateTransaction(event.key, String.valueOf(event.value));
    }

    @Override
    public Map<String, JsonElement> allFlags(LDUser user) {
        FeatureFlagsState state = this.allFlagsState(user, new FlagsStateOption[0]);
        if (!state.isValid()) {
            return null;
        }
        return state.toValuesMap();
    }

    @Override
    public FeatureFlagsState allFlagsState(LDUser user, FlagsStateOption ... options) {
        FeatureFlagsState.Builder builder = new FeatureFlagsState.Builder(options);
        if (this.isOffline()) {
            logger.debug("allFlagsState() was called when client is in offline mode.");
        }
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("allFlagsState() was called before client initialized; using last known values from feature store");
            } else {
                logger.warn("allFlagsState() was called before client initialized; feature store unavailable, returning no data");
                return builder.valid(false).build();
            }
        }
        if (user == null || user.getKey() == null) {
            logger.warn("allFlagsState() was called with null user or null user key! returning no data");
            return builder.valid(false).build();
        }
        boolean clientSideOnly = FlagsStateOption.hasOption(options, FlagsStateOption.CLIENT_SIDE_ONLY);
        Map<String, FeatureFlag> flags = this.featureStore.all(VersionedDataKind.FEATURES);
        for (Map.Entry<String, FeatureFlag> entry : flags.entrySet()) {
            FeatureFlag flag = entry.getValue();
            if (clientSideOnly && !flag.isClientSide()) continue;
            try {
                EvaluationDetail<JsonElement> result = flag.evaluate(user, this.featureStore, EventFactory.DEFAULT).getDetails();
                builder.addFlag(flag, result);
            }
            catch (Exception e) {
                logger.error("Exception caught for feature flag \"{}\" when evaluating all flags: {}", (Object)entry.getKey(), (Object)e.toString());
                logger.debug(e.toString(), (Throwable)e);
                builder.addFlag(entry.getValue(), EvaluationDetail.error(EvaluationReason.ErrorKind.EXCEPTION, null));
            }
        }
        return builder.build();
    }

    @Override
    public boolean boolVariation(String featureKey, LDUser user, boolean defaultValue) {
        return this.evaluate(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive(Boolean.valueOf(defaultValue)), VariationType.Boolean);
    }

    @Override
    public Integer intVariation(String featureKey, LDUser user, int defaultValue) {
        return this.evaluate(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Integer);
    }

    @Override
    public Double doubleVariation(String featureKey, LDUser user, Double defaultValue) {
        return this.evaluate(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Double);
    }

    @Override
    public String stringVariation(String featureKey, LDUser user, String defaultValue) {
        return this.evaluate(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive(defaultValue), VariationType.String);
    }

    @Override
    public JsonElement jsonVariation(String featureKey, LDUser user, JsonElement defaultValue) {
        return this.evaluate(featureKey, user, defaultValue, defaultValue, VariationType.Json);
    }

    @Override
    public EvaluationDetail<Boolean> boolVariationDetail(String featureKey, LDUser user, boolean defaultValue) {
        return this.evaluateDetail(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive(Boolean.valueOf(defaultValue)), VariationType.Boolean, EventFactory.DEFAULT_WITH_REASONS);
    }

    @Override
    public EvaluationDetail<Integer> intVariationDetail(String featureKey, LDUser user, int defaultValue) {
        return this.evaluateDetail(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Integer, EventFactory.DEFAULT_WITH_REASONS);
    }

    @Override
    public EvaluationDetail<Double> doubleVariationDetail(String featureKey, LDUser user, double defaultValue) {
        return this.evaluateDetail(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive((Number)defaultValue), VariationType.Double, EventFactory.DEFAULT_WITH_REASONS);
    }

    @Override
    public EvaluationDetail<String> stringVariationDetail(String featureKey, LDUser user, String defaultValue) {
        return this.evaluateDetail(featureKey, user, defaultValue, (JsonElement)new JsonPrimitive(defaultValue), VariationType.String, EventFactory.DEFAULT_WITH_REASONS);
    }

    @Override
    public EvaluationDetail<JsonElement> jsonVariationDetail(String featureKey, LDUser user, JsonElement defaultValue) {
        return this.evaluateDetail(featureKey, user, defaultValue, defaultValue, VariationType.Json, EventFactory.DEFAULT_WITH_REASONS);
    }

    @Override
    public boolean isFlagKnown(String featureKey) {
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("isFlagKnown called before client initialized for feature flag \"{}\"; using last known values from feature store", (Object)featureKey);
            } else {
                logger.warn("isFlagKnown called before client initialized for feature flag \"{}\"; feature store unavailable, returning false", (Object)featureKey);
                return false;
            }
        }
        try {
            if (this.featureStore.get(VersionedDataKind.FEATURES, featureKey) != null) {
                return true;
            }
        }
        catch (Exception e) {
            logger.error("Encountered exception while calling isFlagKnown for feature flag \"{}\": {}", (Object)e.toString());
            logger.debug(e.toString(), (Throwable)e);
        }
        return false;
    }

    private <T> T evaluate(String featureKey, LDUser user, T defaultValue, JsonElement defaultJson, VariationType<T> expectedType) {
        return this.evaluateDetail(featureKey, user, defaultValue, defaultJson, expectedType, EventFactory.DEFAULT).getValue();
    }

    private <T> EvaluationDetail<T> evaluateDetail(String featureKey, LDUser user, T defaultValue, JsonElement defaultJson, VariationType<T> expectedType, EventFactory eventFactory) {
        EvaluationDetail<JsonElement> details = this.evaluateInternal(featureKey, user, defaultJson, eventFactory);
        Object resultValue = null;
        if (details.getReason().getKind() == EvaluationReason.Kind.ERROR) {
            resultValue = defaultValue;
        } else if (details.getValue() != null) {
            try {
                resultValue = expectedType.coerceValue(details.getValue());
            }
            catch (EvaluationException e) {
                logger.error("Encountered exception in LaunchDarkly client: " + e);
                return EvaluationDetail.error(EvaluationReason.ErrorKind.WRONG_TYPE, defaultValue);
            }
        }
        return new EvaluationDetail<Object>(details.getReason(), details.getVariationIndex(), resultValue);
    }

    private EvaluationDetail<JsonElement> evaluateInternal(String featureKey, LDUser user, JsonElement defaultValue, EventFactory eventFactory) {
        if (!this.initialized()) {
            if (this.featureStore.initialized()) {
                logger.warn("Evaluation called before client initialized for feature flag \"{}\"; using last known values from feature store", (Object)featureKey);
            } else {
                logger.warn("Evaluation called before client initialized for feature flag \"{}\"; feature store unavailable, returning default value", (Object)featureKey);
                this.sendFlagRequestEvent(eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue, EvaluationReason.ErrorKind.CLIENT_NOT_READY));
                return EvaluationDetail.error(EvaluationReason.ErrorKind.CLIENT_NOT_READY, defaultValue);
            }
        }
        FeatureFlag featureFlag = null;
        try {
            featureFlag = this.featureStore.get(VersionedDataKind.FEATURES, featureKey);
            if (featureFlag == null) {
                logger.info("Unknown feature flag \"{}\"; returning default value", (Object)featureKey);
                this.sendFlagRequestEvent(eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue, EvaluationReason.ErrorKind.FLAG_NOT_FOUND));
                return EvaluationDetail.error(EvaluationReason.ErrorKind.FLAG_NOT_FOUND, defaultValue);
            }
            if (user == null || user.getKey() == null) {
                logger.warn("Null user or null user key when evaluating flag \"{}\"; returning default value", (Object)featureKey);
                this.sendFlagRequestEvent(eventFactory.newDefaultFeatureRequestEvent(featureFlag, user, defaultValue, EvaluationReason.ErrorKind.USER_NOT_SPECIFIED));
                return EvaluationDetail.error(EvaluationReason.ErrorKind.USER_NOT_SPECIFIED, defaultValue);
            }
            if (user.getKeyAsString().isEmpty()) {
                logger.warn("User key is blank. Flag evaluation will proceed, but the user will not be stored in LaunchDarkly");
            }
            FeatureFlag.EvalResult evalResult = featureFlag.evaluate(user, this.featureStore, eventFactory);
            for (Event.FeatureRequest event : evalResult.getPrerequisiteEvents()) {
                this.eventProcessor.sendEvent(event);
            }
            EvaluationDetail<JsonElement> details = evalResult.getDetails();
            if (details.isDefaultValue()) {
                details = new EvaluationDetail<JsonElement>(details.getReason(), null, defaultValue);
            }
            this.sendFlagRequestEvent(eventFactory.newFeatureRequestEvent(featureFlag, user, details, defaultValue));
            return details;
        }
        catch (Exception e) {
            logger.error("Encountered exception while evaluating feature flag \"{}\": {}", (Object)featureKey, (Object)e.toString());
            logger.debug(e.toString(), (Throwable)e);
            if (featureFlag == null) {
                this.sendFlagRequestEvent(eventFactory.newUnknownFeatureRequestEvent(featureKey, user, defaultValue, EvaluationReason.ErrorKind.EXCEPTION));
            } else {
                this.sendFlagRequestEvent(eventFactory.newDefaultFeatureRequestEvent(featureFlag, user, defaultValue, EvaluationReason.ErrorKind.EXCEPTION));
            }
            return EvaluationDetail.error(EvaluationReason.ErrorKind.EXCEPTION, defaultValue);
        }
    }

    @Override
    public void close() throws IOException {
        logger.info("Closing LaunchDarkly Client");
        if (this.shouldCloseFeatureStore) {
            this.featureStore.close();
        }
        this.eventProcessor.close();
        this.updateProcessor.close();
    }

    @Override
    public void flush() {
        this.eventProcessor.flush();
    }

    @Override
    public boolean isOffline() {
        return this.config.offline;
    }

    @Override
    public String secureModeHash(LDUser user) {
        if (user == null || user.getKey() == null) {
            return null;
        }
        try {
            Mac mac = Mac.getInstance(HMAC_ALGORITHM);
            mac.init(new SecretKeySpec(this.sdkKey.getBytes(), HMAC_ALGORITHM));
            return Hex.encodeHexString(mac.doFinal(user.getKeyAsString().getBytes("UTF8")));
        }
        catch (UnsupportedEncodingException | InvalidKeyException | NoSuchAlgorithmException e) {
            logger.error("Could not generate secure mode hash: {}", (Object)e.toString());
            logger.debug(e.toString(), (Throwable)e);
            return null;
        }
    }

    @Override
    public String version() {
        return CLIENT_VERSION;
    }

    private static String getClientVersion() {
        Class<LDConfig> clazz = LDConfig.class;
        String className = clazz.getSimpleName() + ".class";
        String classPath = clazz.getResource(className).toString();
        if (!classPath.startsWith("jar")) {
            return "Unknown";
        }
        String manifestPath = classPath.substring(0, classPath.lastIndexOf("!") + 1) + "/META-INF/MANIFEST.MF";
        Manifest manifest = null;
        try {
            manifest = new Manifest(new URL(manifestPath).openStream());
            Attributes attr = manifest.getMainAttributes();
            String value = attr.getValue("Implementation-Version");
            return value;
        }
        catch (IOException e) {
            logger.warn("Unable to determine LaunchDarkly client library version", (Throwable)e);
            return "Unknown";
        }
    }
}

