package com.launchdarkly.client;

import com.google.gson.JsonElement;
import com.launchdarkly.client.value.LDValue;

/**
 * Base class for all analytics events that are generated by the client. Also defines all of its own subclasses.
 */
public class Event {
  final long creationDate;
  final LDUser user;

  public Event(long creationDate, LDUser user) {
    this.creationDate = creationDate;
    this.user = user;
  }
  
  public static final class Custom extends Event {
    final String key;
    final LDValue data;
    final Double metricValue;

    /**
     * @since 4.8.0
     */
    public Custom(long timestamp, String key, LDUser user, LDValue data, Double metricValue) {
      super(timestamp, user);
      this.key = key;
      this.data = data == null ? LDValue.ofNull() : data;
      this.metricValue = metricValue;
    }

    @Deprecated
    public Custom(long timestamp, String key, LDUser user, JsonElement data) {
      this(timestamp, key, user, LDValue.unsafeFromJsonElement(data), null);
    }
  }

  public static final class Identify extends Event {
    public Identify(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }

  public static final class Index extends Event {
    public Index(long timestamp, LDUser user) {
      super(timestamp, user);
    }
  }
  
  public static final class FeatureRequest extends Event {
    final String key;
    final Integer variation;
    final LDValue value;
    final LDValue defaultVal;
    final Integer version;
    final String prereqOf;
    final boolean trackEvents;
    final Long debugEventsUntilDate;
    final EvaluationReason reason;
    final boolean debug;

    /**
     * @since 4.8.0
     */
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, LDValue value,
        LDValue defaultVal, EvaluationReason reason, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, boolean debug) {
      super(timestamp, user);
      this.key = key;
      this.version = version;
      this.variation = variation;
      this.value = value;
      this.defaultVal = defaultVal;
      this.prereqOf = prereqOf;
      this.trackEvents = trackEvents;
      this.debugEventsUntilDate = debugEventsUntilDate;
      this.reason = reason;
      this.debug = debug;
    }
    
    @Deprecated
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, JsonElement value,
        JsonElement defaultVal, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, boolean debug) {
      this(timestamp, key, user, version, variation, LDValue.unsafeFromJsonElement(value), LDValue.unsafeFromJsonElement(defaultVal),
          null, prereqOf, trackEvents, debugEventsUntilDate, debug);
    }

    @Deprecated
    public FeatureRequest(long timestamp, String key, LDUser user, Integer version, Integer variation, JsonElement value,
        JsonElement defaultVal, String prereqOf, boolean trackEvents, Long debugEventsUntilDate, EvaluationReason reason, boolean debug) {
      this(timestamp, key, user, version, variation, LDValue.unsafeFromJsonElement(value), LDValue.unsafeFromJsonElement(defaultVal),
          reason, prereqOf, trackEvents, debugEventsUntilDate, debug);
    }
  }

}