package com.launchdarkly.sdk.server;

import com.google.gson.TypeAdapter;
import com.google.gson.annotations.JsonAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;
import com.launchdarkly.sdk.EvaluationReason;
import com.launchdarkly.sdk.LDValue;
import com.launchdarkly.sdk.json.JsonSerializable;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import static com.launchdarkly.sdk.server.JsonHelpers.gsonInstance;

/**
 * A snapshot of the state of all feature flags with regard to a specific user, generated by
 * calling {@link LDClientInterface#allFlagsState(com.launchdarkly.sdk.LDUser, FlagsStateOption...)}.
 * <p>
 * LaunchDarkly defines a standard JSON encoding for this object, suitable for
 * <a href="https://docs.launchdarkly.com/sdk/client-side/javascript#bootstrapping">bootstrapping</a>
 * the LaunchDarkly JavaScript browser SDK. You can convert it to JSON in any of these ways:
 * <ol>
 * <li> With {@link com.launchdarkly.sdk.json.JsonSerialization}.
 * <li> With Gson, if and only if you configure your {@code Gson} instance with
 * {@link com.launchdarkly.sdk.json.LDGson}.
 * <li> With Jackson, if and only if you configure your {@code ObjectMapper} instance with
 * {@link com.launchdarkly.sdk.json.LDJackson}.
 * </ol>
 * 
 * @since 4.3.0
 */
@JsonAdapter(FeatureFlagsState.JsonSerialization.class)
public class FeatureFlagsState implements JsonSerializable {
  private final Map<String, LDValue> flagValues;
  private final Map<String, FlagMetadata> flagMetadata;
  private final boolean valid;
    
  static class FlagMetadata {
    final Integer variation;
    final EvaluationReason reason;
    final Integer version;
    final Boolean trackEvents;
    final Long debugEventsUntilDate;
    
    FlagMetadata(Integer variation, EvaluationReason reason, Integer version, boolean trackEvents,
        Long debugEventsUntilDate) {
      this.variation = variation;
      this.reason = reason;
      this.version = version;
      this.trackEvents = trackEvents ? Boolean.TRUE : null;
      this.debugEventsUntilDate = debugEventsUntilDate;
    }
    
    @Override
    public boolean equals(Object other) {
      if (other instanceof FlagMetadata) {
        FlagMetadata o = (FlagMetadata)other;
        return Objects.equals(variation, o.variation) &&
            Objects.equals(version, o.version) &&
            Objects.equals(trackEvents, o.trackEvents) &&
            Objects.equals(debugEventsUntilDate, o.debugEventsUntilDate);
      }
      return false;
    }
    
    @Override
    public int hashCode() {
      return Objects.hash(variation, version, trackEvents, debugEventsUntilDate);
    }
  }
  
  private FeatureFlagsState(Map<String, LDValue> flagValues,
      Map<String, FlagMetadata> flagMetadata, boolean valid) {
    this.flagValues = Collections.unmodifiableMap(flagValues);
    this.flagMetadata = Collections.unmodifiableMap(flagMetadata);
    this.valid = valid;
  }
  
  /**
   * Returns true if this object contains a valid snapshot of feature flag state, or false if the
   * state could not be computed (for instance, because the client was offline or there was no user).
   * @return true if the state is valid
   */
  public boolean isValid() {
    return valid;
  }
  
  /**
   * Returns the value of an individual feature flag at the time the state was recorded.
   * @param key the feature flag key
   * @return the flag's JSON value; null if the flag returned the default value, or if there was no such flag
   */
  public LDValue getFlagValue(String key) {
    return flagValues.get(key);
  }

  /**
   * Returns the evaluation reason for an individual feature flag at the time the state was recorded.
   * @param key the feature flag key
   * @return an {@link EvaluationReason}; null if reasons were not recorded, or if there was no such flag
   */
  public EvaluationReason getFlagReason(String key) {
    FlagMetadata data = flagMetadata.get(key);
    return data == null ? null : data.reason;
  }
  
  /**
   * Returns a map of flag keys to flag values. If a flag would have evaluated to the default value,
   * its value will be null.
   * <p>
   * Do not use this method if you are passing data to the front end to "bootstrap" the JavaScript client.
   * Instead, serialize the FeatureFlagsState object to JSON using {@code Gson.toJson()} or {@code Gson.toJsonTree()}.
   * @return an immutable map of flag keys to JSON values
   */
  public Map<String, LDValue> toValuesMap() {
    return flagValues;
  }
  
  @Override
  public boolean equals(Object other) {
    if (other instanceof FeatureFlagsState) {
      FeatureFlagsState o = (FeatureFlagsState)other;
      return flagValues.equals(o.flagValues) &&
          flagMetadata.equals(o.flagMetadata) &&
          valid == o.valid;
    }
    return false;
  }
  
  @Override
  public int hashCode() {
    return Objects.hash(flagValues, flagMetadata, valid);
  }
  
  static class Builder {
    private Map<String, LDValue> flagValues = new HashMap<>();
    private Map<String, FlagMetadata> flagMetadata = new HashMap<>();
    private final boolean saveReasons;
    private final boolean detailsOnlyForTrackedFlags;
    private boolean valid = true;

    Builder(FlagsStateOption... options) {
      saveReasons = FlagsStateOption.hasOption(options, FlagsStateOption.WITH_REASONS);
      detailsOnlyForTrackedFlags = FlagsStateOption.hasOption(options, FlagsStateOption.DETAILS_ONLY_FOR_TRACKED_FLAGS);
    }
    
    Builder valid(boolean valid) {
      this.valid = valid;
      return this;
    }
    
    Builder addFlag(DataModel.FeatureFlag flag, Evaluator.EvalResult eval) {
      flagValues.put(flag.getKey(), eval.getValue());
      final boolean flagIsTracked = flag.isTrackEvents() ||
          (flag.getDebugEventsUntilDate() != null && flag.getDebugEventsUntilDate() > System.currentTimeMillis());
      final boolean wantDetails = !detailsOnlyForTrackedFlags || flagIsTracked;
      FlagMetadata data = new FlagMetadata(
          eval.isDefault() ? null : eval.getVariationIndex(),
          (saveReasons && wantDetails) ? eval.getReason() : null,
          wantDetails ? flag.getVersion() : null,
          flag.isTrackEvents(),
          flag.getDebugEventsUntilDate());
      flagMetadata.put(flag.getKey(), data);
      return this;
    }
    
    FeatureFlagsState build() {
      return new FeatureFlagsState(flagValues, flagMetadata, valid);
    }
  }
  
  static class JsonSerialization extends TypeAdapter<FeatureFlagsState> {
    @Override
    public void write(JsonWriter out, FeatureFlagsState state) throws IOException {
      out.beginObject();
      for (Map.Entry<String, LDValue> entry: state.flagValues.entrySet()) {
        out.name(entry.getKey());
        gsonInstance().toJson(entry.getValue(), LDValue.class, out);
      }
      out.name("$flagsState");
      gsonInstance().toJson(state.flagMetadata, Map.class, out);
      out.name("$valid");
      out.value(state.valid);
      out.endObject();
    }

    // There isn't really a use case for deserializing this, but we have to implement it
    @Override
    public FeatureFlagsState read(JsonReader in) throws IOException {
      Map<String, LDValue> flagValues = new HashMap<>();
      Map<String, FlagMetadata> flagMetadata = new HashMap<>();
      boolean valid = true;
      in.beginObject();
      while (in.hasNext()) {
        String name = in.nextName();
        if (name.equals("$flagsState")) {
          in.beginObject();
          while (in.hasNext()) {
            String metaName = in.nextName();
            FlagMetadata meta = gsonInstance().fromJson(in, FlagMetadata.class);
            flagMetadata.put(metaName, meta);
          }
          in.endObject();
        } else if (name.equals("$valid")) {
          valid = in.nextBoolean();
        } else {
          LDValue value = gsonInstance().fromJson(in, LDValue.class);
          flagValues.put(name, value);
        }
      }
      in.endObject();
      return new FeatureFlagsState(flagValues, flagMetadata, valid);
    }
  }
}
