/*
 * Decompiled with CFR 0.152.
 */
package com.launchdarkly.sdk.server;

import com.launchdarkly.sdk.server.DefaultEventProcessor;
import com.launchdarkly.sdk.server.Util;
import com.launchdarkly.sdk.server.interfaces.BasicConfiguration;
import com.launchdarkly.sdk.server.interfaces.EventSender;
import com.launchdarkly.sdk.server.interfaces.EventSenderFactory;
import com.launchdarkly.sdk.server.interfaces.HttpConfiguration;
import com.launchdarkly.shaded.okhttp3.Headers;
import com.launchdarkly.shaded.okhttp3.MediaType;
import com.launchdarkly.shaded.okhttp3.OkHttpClient;
import com.launchdarkly.shaded.okhttp3.Request;
import com.launchdarkly.shaded.okhttp3.RequestBody;
import com.launchdarkly.shaded.okhttp3.Response;
import java.io.IOException;
import java.net.URI;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Date;
import java.util.UUID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class DefaultEventSender
implements EventSender {
    private static final Logger logger = LoggerFactory.getLogger(DefaultEventProcessor.class);
    static final Duration DEFAULT_RETRY_DELAY = Duration.ofSeconds(1L);
    private static final String EVENT_SCHEMA_HEADER = "X-LaunchDarkly-Event-Schema";
    private static final String EVENT_SCHEMA_VERSION = "3";
    private static final String EVENT_PAYLOAD_ID_HEADER = "X-LaunchDarkly-Payload-ID";
    private static final MediaType JSON_CONTENT_TYPE = MediaType.parse("application/json; charset=utf-8");
    private static final SimpleDateFormat HTTP_DATE_FORMAT = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss zzz");
    private static final Object HTTP_DATE_FORMAT_LOCK = new Object();
    private final OkHttpClient httpClient;
    private final Headers baseHeaders;
    final Duration retryDelay;

    DefaultEventSender(HttpConfiguration httpConfiguration, Duration retryDelay) {
        OkHttpClient.Builder httpBuilder = new OkHttpClient.Builder();
        Util.configureHttpClientBuilder(httpConfiguration, httpBuilder);
        this.httpClient = httpBuilder.build();
        this.baseHeaders = Util.getHeadersBuilderFor(httpConfiguration).add("Content-Type", "application/json").build();
        this.retryDelay = retryDelay == null ? DEFAULT_RETRY_DELAY : retryDelay;
    }

    @Override
    public void close() throws IOException {
        Util.shutdownHttpClient(this.httpClient);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public EventSender.Result sendEventData(EventSender.EventDataKind kind, String data, int eventCount, URI eventsBaseUri) {
        String description;
        String path;
        if (data == null) return new EventSender.Result(true, false, null);
        if (data.isEmpty()) {
            return new EventSender.Result(true, false, null);
        }
        Headers.Builder headersBuilder = this.baseHeaders.newBuilder();
        switch (kind) {
            case ANALYTICS: {
                path = "bulk";
                String eventPayloadId = UUID.randomUUID().toString();
                headersBuilder.add(EVENT_PAYLOAD_ID_HEADER, eventPayloadId);
                headersBuilder.add(EVENT_SCHEMA_HEADER, EVENT_SCHEMA_VERSION);
                description = String.format("%d event(s)", eventCount);
                break;
            }
            case DIAGNOSTICS: {
                path = "diagnostic";
                description = "diagnostic event";
                break;
            }
            default: {
                throw new IllegalArgumentException("kind");
            }
        }
        URI uri = eventsBaseUri.resolve(eventsBaseUri.getPath().endsWith("/") ? path : "/" + path);
        Headers headers = headersBuilder.build();
        RequestBody body = RequestBody.create(data, JSON_CONTENT_TYPE);
        boolean mustShutDown = false;
        logger.debug("Posting {} to {} with payload: {}", new Object[]{description, uri, data});
        int attempt = 0;
        while (attempt < 2) {
            if (attempt > 0) {
                logger.warn("Will retry posting {} after {}", (Object)description, (Object)Util.describeDuration(this.retryDelay));
                try {
                    Thread.sleep(this.retryDelay.toMillis());
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
            Request request = new Request.Builder().url(uri.toASCIIString()).post(body).headers(headers).build();
            long startTime = System.currentTimeMillis();
            String nextActionMessage = attempt == 0 ? "will retry" : "some events were dropped";
            String errorContext = "posting " + description;
            try (Response response = this.httpClient.newCall(request).execute();){
                long endTime = System.currentTimeMillis();
                logger.debug("{} delivery took {} ms, response status {}", new Object[]{description, endTime - startTime, response.code()});
                if (response.isSuccessful()) {
                    EventSender.Result result = new EventSender.Result(true, false, DefaultEventSender.parseResponseDate(response));
                    return result;
                }
                String errorDesc = Util.httpErrorDescription(response.code());
                boolean recoverable = Util.checkIfErrorIsRecoverableAndLog(logger, errorDesc, errorContext, response.code(), nextActionMessage);
                if (!recoverable) {
                    mustShutDown = true;
                    return new EventSender.Result(false, mustShutDown, null);
                }
            }
            catch (IOException e) {
                Util.checkIfErrorIsRecoverableAndLog(logger, e.toString(), errorContext, 0, nextActionMessage);
            }
            ++attempt;
        }
        return new EventSender.Result(false, mustShutDown, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static final Date parseResponseDate(Response response) {
        String dateStr = response.header("Date");
        if (dateStr != null) {
            try {
                Object object = HTTP_DATE_FORMAT_LOCK;
                synchronized (object) {
                    return HTTP_DATE_FORMAT.parse(dateStr);
                }
            }
            catch (ParseException e) {
                logger.warn("Received invalid Date header from events service");
            }
        }
        return null;
    }

    static final class Factory
    implements EventSenderFactory {
        Factory() {
        }

        @Override
        public EventSender createEventSender(BasicConfiguration basicConfiguration, HttpConfiguration httpConfiguration) {
            return new DefaultEventSender(httpConfiguration, DEFAULT_RETRY_DELAY);
        }
    }
}

