/**
 * Copyright (c) 2013-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 * @typechecks
 * 
 */

/**
 * Stateful API for text direction detection
 *
 * This class can be used in applications where you need to detect the
 * direction of a sequence of text blocks, where each direction shall be used
 * as the fallback direction for the next one.
 *
 * NOTE: A default direction, if not provided, is set based on the global
 *       direction, as defined by `UnicodeBidiDirection`.
 *
 * == Example ==
 * ```
 * var UnicodeBidiService = require('UnicodeBidiService');
 *
 * var bidiService = new UnicodeBidiService();
 *
 * ...
 *
 * bidiService.reset();
 * for (var para in paragraphs) {
 *   var dir = bidiService.getDirection(para);
 *   ...
 * }
 * ```
 *
 * Part of our implementation of Unicode Bidirectional Algorithm (UBA)
 * Unicode Standard Annex #9 (UAX9)
 * http://www.unicode.org/reports/tr9/
 */

'use strict';

Liferay.Loader.define('frontend-js-tooltip-support-web$fbjs@0.8.17/lib/UnicodeBidiService', ['module', 'exports', 'require', './UnicodeBidi', './UnicodeBidiDirection', './invariant'], function (module, exports, require) {
  var define = undefined;
  function _classCallCheck(instance, Constructor) {
    if (!(instance instanceof Constructor)) {
      throw new TypeError("Cannot call a class as a function");
    }
  }

  var UnicodeBidi = require('./UnicodeBidi');
  var UnicodeBidiDirection = require('./UnicodeBidiDirection');

  var invariant = require('./invariant');

  var UnicodeBidiService = function () {

    /**
     * Stateful class for paragraph direction detection
     *
     * @param defaultDir  Default direction of the service
     */
    function UnicodeBidiService(defaultDir) {
      _classCallCheck(this, UnicodeBidiService);

      if (!defaultDir) {
        defaultDir = UnicodeBidiDirection.getGlobalDir();
      } else {
        !UnicodeBidiDirection.isStrong(defaultDir) ? invariant(false) : void 0;
      }
      this._defaultDir = defaultDir;
      this.reset();
    }

    /**
     * Reset the internal state
     *
     * Instead of creating a new instance, you can just reset() your instance
     * everytime you start a new loop.
     */

    UnicodeBidiService.prototype.reset = function reset() {
      this._lastDir = this._defaultDir;
    };

    /**
     * Returns the direction of a block of text, and remembers it as the
     * fall-back direction for the next paragraph.
     *
     * @param str  A text block, e.g. paragraph, table cell, tag
     * @return     The resolved direction
     */

    UnicodeBidiService.prototype.getDirection = function getDirection(str) {
      this._lastDir = UnicodeBidi.getDirection(str, this._lastDir);
      return this._lastDir;
    };

    return UnicodeBidiService;
  }();

  module.exports = UnicodeBidiService;
});
//# sourceMappingURL=UnicodeBidiService.js.map