/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import {openToast, throttle} from 'frontend-js-web';
import React, {
	useCallback,
	useContext,
	useEffect,
	useMemo,
	useReducer,
	useRef,
	useState,
} from 'react';
import {useDrag, useDrop} from 'react-dnd';
import {getEmptyImage} from 'react-dnd-html5-backend';

import {FRAGMENT_ENTRY_TYPES} from '../../config/constants/fragmentEntryTypes';
import {LAYOUT_DATA_ITEM_TYPES} from '../../config/constants/layoutDataItemTypes';
import {config} from '../../config/index';
import {
	useCollectionItemIndex,
	useParentToControlsId,
	useToControlsId,
} from '../../contexts/CollectionItemContext';
import {useSelectItem} from '../../contexts/ControlsContext';
import {useSelectorRef} from '../../contexts/StoreContext';
import {formIsMapped} from '../formIsMapped';
import {hasFormParent} from '../hasFormParent';
import {DRAG_DROP_TARGET_TYPE} from './constants/dragDropTargetType';
import defaultComputeHover from './defaultComputeHover';
import getDropData from './getDropData';

export const initialDragDrop = {
	canDrag: true,

	dispatch: null,

	layoutDataRef: {
		current: {
			items: [],
		},
	},

	setCanDrag: () => {},

	state: {

		/**
		 * Item that is being dragged
		 */
		dropItem: null,

		/**
		 * Target item where the item is being dragged true.
		 * If elevate is true, dropTargetItem is the sibling
		 * of drop item, otherwise is it's parent.
		 */
		dropTargetItem: null,

		/**
		 * When false, an "invalid drop" advise should be shown
		 * to users.
		 */
		droppable: true,

		/**
		 * If true, dropTargetItem is the sibling of dropItem
		 * and targetPosition determines the item index.
		 */
		elevate: false,

		/**
		 * Vertical position relative to dropTargetItem
		 * (bottom, middle, top)
		 */
		targetPositionWithMiddle: null,

		/**
		 * Vertical position relative to dropTargetItem
		 * (bottom, top)
		 */
		targetPositionWithoutMiddle: null,

		/**
		 * Source of the Drag and Drop status
		 * (where the drag and drop status have been generated)
		 */
		type: DRAG_DROP_TARGET_TYPE.INITIAL,
	},

	targetRefs: new Map(),
};

const DragAndDropContext = React.createContext(initialDragDrop);

export function useDropTargetData() {
	const {dropTargetItem, targetPositionWithMiddle} = useContext(
		DragAndDropContext
	).state;

	return {
		item: dropTargetItem,
		position: targetPositionWithMiddle,
	};
}

export function useIsDroppable() {
	return useContext(DragAndDropContext).state.droppable;
}

export function useSetCanDrag() {
	return useContext(DragAndDropContext).setCanDrag;
}

export function NotDraggableArea({children}) {
	return (
		<div
			draggable
			onDragStart={(event) => {
				event.preventDefault();
				event.stopPropagation();
			}}
		>
			{children}
		</div>
	);
}

export function useDragItem(sourceItem, onDragEnd, onBegin = () => {}) {
	const {canDrag, dispatch, layoutDataRef, state} = useContext(
		DragAndDropContext
	);
	const sourceRef = useRef(null);

	const item = {
		...sourceItem,
		id: sourceItem.itemId,
		namespace: config.portletNamespace,
	};

	if (!sourceItem.origin) {
		delete item.origin;
	}

	const [{isDraggingSource}, handlerRef, previewRef] = useDrag({
		begin() {
			onBegin();
		},

		canDrag,

		collect: (monitor) => ({
			isDraggingSource: monitor.isDragging(),
		}),

		end() {
			computeDrop({
				dispatch,
				layoutDataRef,
				onDragEnd,
				state,
			});
		},

		item,
	});

	useEffect(() => {
		previewRef(getEmptyImage(), {captureDraggingState: true});
	}, [previewRef]);

	return {
		handlerRef,
		isDraggingSource,
		sourceRef,
	};
}

export function useDragSymbol(
	{fragmentEntryType, icon, isWidget, label, type},
	onDragEnd
) {
	const selectItem = useSelectItem();

	const sourceItem = useMemo(
		() => ({
			fragmentEntryType,
			icon,
			isSymbol: true,
			isWidget,
			itemId: label,
			name: label,
			type,
		}),
		[fragmentEntryType, icon, isWidget, label, type]
	);

	const {handlerRef, isDraggingSource, sourceRef} = useDragItem(
		sourceItem,
		onDragEnd,
		() => selectItem(null)
	);

	const symbolRef = (element) => {
		sourceRef.current = element;
		handlerRef(element);
	};

	return {
		isDraggingSource,
		sourceRef: symbolRef,
	};
}

export function useDropClear() {
	const {dispatch} = useContext(DragAndDropContext);

	const [, dropClearRef] = useDrop({
		accept: Object.values(LAYOUT_DATA_ITEM_TYPES),
		hover() {
			dispatch(initialDragDrop.state);
		},
	});

	return dropClearRef;
}

export function useDropTarget(_targetItem, computeHover = defaultComputeHover) {
	const collectionItemIndex = useCollectionItemIndex();
	const toControlsId = useToControlsId();
	const parentToControlsId = useParentToControlsId();

	const {dispatch, layoutDataRef, state, targetRefs} = useContext(
		DragAndDropContext
	);
	const targetRef = useRef(null);

	const targetItem = useMemo(
		() => ({
			..._targetItem,
			collectionItemIndex,
			parentToControlsId,
			toControlsId,
		}),
		[_targetItem, collectionItemIndex, toControlsId, parentToControlsId]
	);

	const isOverTarget =
		state.dropTargetItem &&
		targetItem &&
		state.dropTargetItem.toControlsId(state.dropTargetItem.itemId) ===
			targetItem.toControlsId(targetItem.itemId);

	const [, setDropTargetRef] = useDrop({
		accept: Object.values(LAYOUT_DATA_ITEM_TYPES),
		hover(source, monitor) {
			if (source.origin !== targetItem.origin) {
				return;
			}
			computeHover({
				dispatch,
				layoutDataRef,
				monitor,
				sourceItem: source,
				targetItem,
				targetRefs,
				toControlsId,
			});
		},
	});

	useEffect(() => {
		const itemId = toControlsId(targetItem.itemId);

		targetRefs.set(itemId, targetRef);

		return () => {
			targetRefs.delete(itemId);
		};
	}, [layoutDataRef, targetItem, targetRef, targetRefs, toControlsId]);

	const setTargetRef = useCallback(
		(element) => {
			setDropTargetRef(element);

			targetRef.current = element;
		},
		[setDropTargetRef]
	);

	return {
		isOverTarget,
		sourceItem: state.dropItem,
		targetItemId: state.dropTargetItem?.toControlsId(
			state.dropTargetItem.itemId
		),
		targetPosition: state.targetPositionWithMiddle,
		targetRef: setTargetRef,
	};
}

export function DragAndDropContextProvider({children}) {
	const [canDrag, setCanDrag] = useState(true);

	const [state, reducerDispatch] = useReducer(
		(state, nextState) =>
			Object.keys(state).some((key) => state[key] !== nextState[key])
				? nextState
				: state,
		initialDragDrop.state
	);

	const targetRefs = useMemo(() => new Map(), []);

	const dispatch = useMemo(() => {
		return throttle(reducerDispatch, 100);
	}, [reducerDispatch]);

	const layoutDataRef = useSelectorRef((state) => state.layoutData);

	const dragAndDropContext = useMemo(
		() => ({
			canDrag,
			dispatch,
			layoutDataRef,
			setCanDrag,
			state,
			targetRefs,
		}),
		[canDrag, dispatch, layoutDataRef, state, targetRefs, setCanDrag]
	);

	return (
		<DragAndDropContext.Provider value={dragAndDropContext}>
			{children}
		</DragAndDropContext.Provider>
	);
}

function computeDrop({dispatch, layoutDataRef, onDragEnd, state}) {
	if (!state.droppable) {
		let message = '';

		if (state.dropTargetItem.type === LAYOUT_DATA_ITEM_TYPES.dropZone) {
			message = Liferay.Language.get(
				'fragments-and-widgets-cannot-be-placed-inside-this-area'
			);
		}
		else if (
			state.dropTargetItem.type === LAYOUT_DATA_ITEM_TYPES.collection
		) {
			message = Liferay.Language.get(
				'fragments-cannot-be-placed-inside-an-unmapped-collection-display-fragment'
			);
		}
		else if (
			state.dropTargetItem.type === LAYOUT_DATA_ITEM_TYPES.form &&
			!formIsMapped(state.dropTargetItem)
		) {
			message = Liferay.Language.get(
				'fragments-cannot-be-placed-inside-an-unmapped-form-container'
			);
		}
		else if (
			state.dropItem.fragmentEntryType === FRAGMENT_ENTRY_TYPES.input
		) {
			message = Liferay.Language.get(
				'form-components-can-only-be-placed-inside-a-mapped-form-container'
			);
		}
		else if (
			state.dropItem.isWidget &&
			hasFormParent(state.dropItem, layoutDataRef.current)
		) {
			message = Liferay.Language.get(
				'widgets-cannot-be-placed-inside-a-form-container'
			);
		}
		else if (state.dropItem.parentId !== state.dropTargetItem.itemId) {
			message = Liferay.Language.get('an-unexpected-error-occurred');
		}

		if (message) {
			openToast({
				message,
				type: 'danger',
			});
		}

		dispatch(initialDragDrop.state);

		return;
	}

	if (state.dropItem && state.dropTargetItem) {
		const {dropItemId, position} = getDropData({
			isElevation: state.elevate,
			layoutDataRef,
			sourceItemId: state.dropItem.itemId,
			targetItemId: state.dropTargetItem.itemId,
			targetPosition: state.targetPositionWithoutMiddle,
		});

		onDragEnd(dropItemId, position);
	}

	dispatch(initialDragDrop.state);
}
