/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.workflow.kaleo.runtime.internal.util;

import com.liferay.portal.kernel.exception.PortalException;
import com.liferay.portal.kernel.json.JSONFactory;
import com.liferay.portal.kernel.json.JSONObject;
import com.liferay.portal.kernel.service.ServiceContext;
import com.liferay.portal.workflow.kaleo.model.KaleoInstance;
import com.liferay.portal.workflow.kaleo.model.KaleoInstanceToken;
import com.liferay.portal.workflow.kaleo.model.KaleoTaskInstanceToken;
import com.liferay.portal.workflow.kaleo.model.KaleoTimerInstanceToken;
import com.liferay.portal.workflow.kaleo.runtime.ExecutionContext;
import com.liferay.portal.workflow.kaleo.runtime.util.ExecutionContextHelper;
import com.liferay.portal.workflow.kaleo.runtime.util.WorkflowContextUtil;
import com.liferay.portal.workflow.kaleo.service.KaleoInstanceTokenLocalService;
import com.liferay.portal.workflow.kaleo.service.KaleoLogLocalService;
import com.liferay.portal.workflow.kaleo.service.KaleoTaskInstanceTokenLocalService;
import com.liferay.portal.workflow.kaleo.service.KaleoTimerInstanceTokenLocalService;

import java.io.Serializable;

import java.util.List;
import java.util.Map;

import org.osgi.service.component.annotations.Component;
import org.osgi.service.component.annotations.Reference;

/**
 * @author Marcellus Tavares
 */
@Component(service = ExecutionContextHelper.class)
public class ExecutionContextHelperImpl implements ExecutionContextHelper {

	@Override
	public void checkKaleoInstanceComplete(ExecutionContext executionContext)
		throws PortalException {

		KaleoInstanceToken kaleoInstanceToken =
			executionContext.getKaleoInstanceToken();

		if (!kaleoInstanceToken.isCompleted()) {
			return;
		}

		KaleoInstance kaleoInstance = kaleoInstanceToken.getKaleoInstance();

		if (!kaleoInstance.isCompleted()) {
			return;
		}

		_kaleoLogLocalService.addWorkflowInstanceEndKaleoLog(
			kaleoInstanceToken, executionContext.getServiceContext());
	}

	@Override
	public void completeKaleoTimerInstances(ExecutionContext executionContext)
		throws PortalException {

		KaleoInstanceToken kaleoInstanceToken =
			executionContext.getKaleoInstanceToken();

		List<KaleoTimerInstanceToken> kaleoTimerInstanceTokens =
			_kaleoTimerInstanceTokenLocalService.getKaleoTimerInstanceTokens(
				kaleoInstanceToken.getKaleoInstanceTokenId(), false, false,
				executionContext.getServiceContext());

		_kaleoTimerInstanceTokenLocalService.completeKaleoTimerInstanceTokens(
			kaleoTimerInstanceTokens, executionContext.getServiceContext());
	}

	@Override
	public String convert(ExecutionContext executionContext) {
		JSONObject jsonObject = _jsonFactory.createJSONObject();

		KaleoInstanceToken kaleoInstanceToken =
			executionContext.getKaleoInstanceToken();

		if (kaleoInstanceToken != null) {
			jsonObject.put(
				"kaleoInstanceTokenId",
				kaleoInstanceToken.getKaleoInstanceTokenId());
		}

		KaleoTaskInstanceToken kaleoTaskInstanceToken =
			executionContext.getKaleoTaskInstanceToken();

		if (kaleoTaskInstanceToken != null) {
			jsonObject.put(
				"kaleoTaskInstanceTokenId",
				kaleoTaskInstanceToken.getKaleoTaskInstanceTokenId());
		}

		jsonObject.put(
			"serviceContext",
			_jsonFactory.serialize(executionContext.getServiceContext())
		).put(
			"transitionName", executionContext.getTransitionName()
		).put(
			"workflowContext",
			WorkflowContextUtil.convert(executionContext.getWorkflowContext())
		);

		return jsonObject.toString();
	}

	@Override
	public ExecutionContext convert(String json) throws Exception {
		JSONObject jsonObject = _jsonFactory.createJSONObject(json);

		KaleoInstanceToken kaleoInstanceToken = null;

		long kaleoInstanceTokenId = jsonObject.getLong("kaleoInstanceTokenId");

		if (kaleoInstanceTokenId > 0) {
			kaleoInstanceToken =
				_kaleoInstanceTokenLocalService.getKaleoInstanceToken(
					kaleoInstanceTokenId);
		}

		KaleoTaskInstanceToken kaleoTaskInstanceToken = null;

		long kaleoTaskInstanceTokenId = jsonObject.getLong(
			"kaleoTaskInstanceTokenId");

		if (kaleoTaskInstanceTokenId > 0) {
			kaleoTaskInstanceToken =
				_kaleoTaskInstanceTokenLocalService.getKaleoTaskInstanceToken(
					kaleoTaskInstanceTokenId);
		}

		Map<String, Serializable> workflowContext = WorkflowContextUtil.convert(
			jsonObject.getString("workflowContext"));

		ServiceContext serviceContext =
			(ServiceContext)_jsonFactory.deserialize(
				jsonObject.getString("serviceContext"));

		return new ExecutionContext(
			kaleoInstanceToken, kaleoTaskInstanceToken, workflowContext,
			serviceContext);
	}

	@Override
	public boolean isKaleoInstanceBlocked(ExecutionContext executionContext) {
		KaleoInstanceToken kaleoInstanceToken =
			executionContext.getKaleoInstanceToken();

		int count =
			_kaleoTimerInstanceTokenLocalService.
				getKaleoTimerInstanceTokensCount(
					kaleoInstanceToken.getKaleoInstanceTokenId(), true, false,
					executionContext.getServiceContext());

		if (count > 0) {
			return true;
		}

		return false;
	}

	@Reference
	private JSONFactory _jsonFactory;

	@Reference
	private KaleoInstanceTokenLocalService _kaleoInstanceTokenLocalService;

	@Reference
	private KaleoLogLocalService _kaleoLogLocalService;

	@Reference
	private KaleoTaskInstanceTokenLocalService
		_kaleoTaskInstanceTokenLocalService;

	@Reference
	private KaleoTimerInstanceTokenLocalService
		_kaleoTimerInstanceTokenLocalService;

}