/*
 * Copyright 2007 Guy Van den Broeck
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.outerj.daisy.diff.html.ancestor;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.compare.internal.LCSSettings;
import org.eclipse.compare.rangedifferencer.IRangeComparator;
import org.eclipse.compare.rangedifferencer.RangeDifference;
import org.eclipse.compare.rangedifferencer.RangeDifferencer;
import org.eclipse.core.runtime.IProgressMonitor;
import org.outerj.daisy.diff.html.dom.Node;
import org.outerj.daisy.diff.html.dom.TagNode;
import org.outerj.daisy.diff.html.dom.TextNode;

/**
 * A comparator that compares only the elements of text inside a given tag.
 */
public final class TextOnlyComparator implements IRangeComparator {

    private TextNode[] leafs;

    public TextOnlyComparator(TagNode tree) {
        List<TextNode> leafs = new ArrayList<TextNode>();
        addRecursive(tree, leafs);
        this.leafs = leafs.toArray(new TextNode[leafs.size()]);
    }

    private static void addRecursive(TagNode tree, List<TextNode> leafs) {
        for (Node child : tree) {
            if (child instanceof TagNode) {
                TagNode tagnode = (TagNode) child;
                addRecursive(tagnode, leafs);
            } else if (child instanceof TextNode) {
                TextNode textnode = (TextNode) child;
                leafs.add(textnode);
            }
        }
    }

    public int getRangeCount() {
        return leafs.length;
    }

    public boolean rangesEqual(int owni, IRangeComparator otherComp, int otheri) {
        TextOnlyComparator other;
        try {
            other = (TextOnlyComparator) otherComp;
        } catch (ClassCastException e) {
            return false;
        }

        return leafs[owni].isSameText(other.leafs[otheri]);
    }

    public boolean skipRangeComparison(int arg0, int arg1, IRangeComparator arg2) {
        return false;
    }

    public double getMatchRatio(TextOnlyComparator other, IProgressMonitor progressMonitor) {
        LCSSettings settings = new LCSSettings();
        settings.setUseGreedyMethod(true);
        settings.setPowLimit(1.5);
        settings.setTooLong(150 * 150);

        RangeDifference[] differences = RangeDifferencer.findDifferences(
                progressMonitor, settings, other, this);
        int distanceOther = 0;
        for (RangeDifference d : differences) {
            distanceOther += d.leftLength();
        }

        int distanceThis = 0;
        for (RangeDifference d : differences) {
            distanceThis += d.rightLength();
        }

        return ((0.0 + distanceOther) / other.getRangeCount() + (0.0 + distanceThis)
                / getRangeCount()) / 2;
    }
}
