/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 */

package com.liferay.item.selector.taglib.internal.display.context;

import com.liferay.item.selector.ItemSelector;
import com.liferay.item.selector.ItemSelectorCriterion;
import com.liferay.item.selector.provider.GroupItemSelectorProvider;
import com.liferay.item.selector.taglib.internal.servlet.item.selector.ItemSelectorUtil;
import com.liferay.item.selector.taglib.internal.util.EntryURLUtil;
import com.liferay.item.selector.taglib.internal.util.GroupItemSelectorProviderRegistryUtil;
import com.liferay.petra.string.StringPool;
import com.liferay.portal.kernel.dao.search.SearchContainer;
import com.liferay.portal.kernel.model.Group;
import com.liferay.portal.kernel.portlet.LiferayPortletRequest;
import com.liferay.portal.kernel.portlet.RequestBackedPortletURLFactoryUtil;
import com.liferay.portal.kernel.portlet.url.builder.PortletURLBuilder;
import com.liferay.portal.kernel.theme.ThemeDisplay;
import com.liferay.portal.kernel.util.GetterUtil;
import com.liferay.portal.kernel.util.ParamUtil;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portlet.usersadmin.search.GroupSearch;

import java.util.List;
import java.util.Set;

import javax.portlet.PortletURL;

/**
 * @author Cristina González
 */
public class GroupSelectorDisplayContext {

	public GroupSelectorDisplayContext(
		LiferayPortletRequest liferayPortletRequest) {

		_liferayPortletRequest = liferayPortletRequest;
	}

	public String getGroupItemSelectorIcon() {
		GroupItemSelectorProvider groupItemSelectorProvider =
			GroupItemSelectorProviderRegistryUtil.getGroupItemSelectorProvider(
				_getGroupType());

		if (groupItemSelectorProvider == null) {
			return "folder";
		}

		String icon = groupItemSelectorProvider.getIcon();

		if (icon == null) {
			return "folder";
		}

		return icon;
	}

	public String getGroupItemSelectorLabel(String groupType) {
		GroupItemSelectorProvider groupItemSelectorProvider =
			GroupItemSelectorProviderRegistryUtil.getGroupItemSelectorProvider(
				groupType);

		if (groupItemSelectorProvider == null) {
			return StringPool.BLANK;
		}

		String label = groupItemSelectorProvider.getLabel(
			_liferayPortletRequest.getLocale());

		if (label == null) {
			return StringPool.BLANK;
		}

		return label;
	}

	public PortletURL getGroupItemSelectorURL(String groupType) {
		return PortletURLBuilder.create(
			_getItemSelectorURL()
		).setParameter(
			"groupType", groupType
		).setParameter(
			"scopeGroupType",
			ParamUtil.getString(_liferayPortletRequest, "scopeGroupType")
		).setParameter(
			"selectedTab",
			ParamUtil.getString(_liferayPortletRequest, "selectedTab")
		).setParameter(
			"showGroupSelector", true
		).buildPortletURL();
	}

	public Set<String> getGroupTypes() {
		return GroupItemSelectorProviderRegistryUtil.
			getGroupItemSelectorProviderTypes();
	}

	public SearchContainer<Group> getSearchContainer() {
		SearchContainer<Group> searchContainer = new GroupSearch(
			_liferayPortletRequest, _getIteratorURL());

		searchContainer.setEmptyResultsMessage(_getEmptyResultsMessage());
		searchContainer.setResultsAndTotal(
			() -> (List<Group>)_liferayPortletRequest.getAttribute(
				"liferay-item-selector:group-selector:groups"),
			GetterUtil.getInteger(
				_liferayPortletRequest.getAttribute(
					"liferay-item-selector:group-selector:groupsCount")));

		return searchContainer;
	}

	public String getViewGroupURL(Group group) {
		PortletURL portletURL = EntryURLUtil.getGroupPortletURL(
			group, _liferayPortletRequest);

		return portletURL.toString();
	}

	public boolean isGroupTypeActive(String groupType) {
		if (groupType.equals(_getGroupType())) {
			return true;
		}

		return false;
	}

	protected GroupSelectorDisplayContext(
		String groupType, LiferayPortletRequest liferayPortletRequest) {

		_groupType = groupType;
		_liferayPortletRequest = liferayPortletRequest;
	}

	private String _getEmptyResultsMessage() {
		GroupItemSelectorProvider groupItemSelectorProvider =
			GroupItemSelectorProviderRegistryUtil.getGroupItemSelectorProvider(
				_getGroupType());

		if (groupItemSelectorProvider == null) {
			return GroupSearch.EMPTY_RESULTS_MESSAGE;
		}

		ThemeDisplay themeDisplay =
			(ThemeDisplay)_liferayPortletRequest.getAttribute(
				WebKeys.THEME_DISPLAY);

		String emptyResultsMessage =
			groupItemSelectorProvider.getEmptyResultsMessage(
				themeDisplay.getLocale());

		if (emptyResultsMessage == null) {
			return GroupSearch.EMPTY_RESULTS_MESSAGE;
		}

		return emptyResultsMessage;
	}

	private String _getGroupType() {
		if (_groupType != null) {
			return _groupType;
		}

		_groupType = ParamUtil.getString(_liferayPortletRequest, "groupType");

		return _groupType;
	}

	private ItemSelector _getItemSelector() {
		return ItemSelectorUtil.getItemSelector();
	}

	private PortletURL _getItemSelectorURL() {
		ItemSelector itemSelector = _getItemSelector();

		List<ItemSelectorCriterion> itemSelectorCriteria =
			itemSelector.getItemSelectorCriteria(
				_liferayPortletRequest.getParameterMap());

		return itemSelector.getItemSelectorURL(
			RequestBackedPortletURLFactoryUtil.create(_liferayPortletRequest),
			ParamUtil.getString(
				_liferayPortletRequest, "itemSelectedEventName"),
			itemSelectorCriteria.toArray(new ItemSelectorCriterion[0]));
	}

	private PortletURL _getIteratorURL() {
		return PortletURLBuilder.create(
			_getItemSelectorURL()
		).setParameter(
			"groupType", _getGroupType()
		).setParameter(
			"selectedTab",
			ParamUtil.getString(_liferayPortletRequest, "selectedTab")
		).setParameter(
			"showGroupSelector", true
		).buildPortletURL();
	}

	private String _groupType;
	private final LiferayPortletRequest _liferayPortletRequest;

}