/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

package com.liferay.portal.search.test.util.mappings;

import com.liferay.portal.kernel.search.Document;
import com.liferay.portal.kernel.search.Field;
import com.liferay.portal.kernel.search.Hits;
import com.liferay.portal.kernel.search.Query;
import com.liferay.portal.kernel.search.filter.TermFilter;
import com.liferay.portal.search.analysis.FieldQueryBuilder;
import com.liferay.portal.search.test.util.DocumentsAssert;
import com.liferay.portal.search.test.util.indexing.BaseIndexingTestCase;
import com.liferay.portal.search.test.util.indexing.DocumentCreationHelpers;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.junit.Assert;

/**
 * @author André de Oliveira
 */
public abstract class BaseFieldQueryBuilderTestCase
	extends BaseIndexingTestCase {

	protected void addDocument(String... values) throws Exception {
		String[] transformed = transformFieldValues(values);

		if (transformed != null) {
			values = transformed;
		}

		addDocument(DocumentCreationHelpers.singleText(getField(), values));

		String[] values2 = values;

		assertSearch(
			indexingTestHelper -> {
				String keywords = values2[0];

				prepareSearch(indexingTestHelper, keywords);

				indexingTestHelper.search();

				indexingTestHelper.verify(
					hits -> {
						Document[] documents = hits.getDocs();

						String field = getField();

						List<String> expectedValues = Arrays.asList(values2);

						List<String> actualValues = new ArrayList<>();

						for (Document document : documents) {
							List<String> documentValues = Arrays.asList(
								document.getValues(field));

							if (documentValues.equals(expectedValues)) {
								return;
							}

							actualValues.addAll(documentValues);
						}

						Assert.assertEquals(
							keywords + "->" + actualValues,
							expectedValues.toString(), actualValues.toString());
					});
			});
	}

	protected void assertSearch(String keywords, int size) throws Exception {
		_assertCount(keywords, size);
	}

	protected void assertSearch(String keywords, List<String> values)
		throws Exception {

		assertSearch(
			indexingTestHelper -> {
				prepareSearch(indexingTestHelper, keywords);

				indexingTestHelper.search();

				indexingTestHelper.verify(
					hits -> DocumentsAssert.assertValues(
						keywords, hits.getDocs(), getField(), values));
			});
	}

	protected void assertSearchCount(String keywords, int size)
		throws Exception {

		assertSearch(
			indexingTestHelper -> {
				prepareSearch(indexingTestHelper, keywords);

				long count = indexingTestHelper.searchCount();

				Assert.assertEquals(keywords, size, count);
			});
	}

	protected void assertSearchIgnoreRelevance(
			String keywords, List<String> values)
		throws Exception {

		assertSearch(
			indexingTestHelper -> {
				prepareSearch(indexingTestHelper, keywords);

				indexingTestHelper.search();

				indexingTestHelper.verify(
					hits -> DocumentsAssert.assertValuesIgnoreRelevance(
						keywords, hits.getDocs(), getField(), values));
			});
	}

	protected void assertSearchNoHits(String keywords) throws Exception {
		_assertCount(keywords, 0);
	}

	protected Query buildQuery(String keywords) {
		FieldQueryBuilder fieldQueryBuilder = createFieldQueryBuilder();

		Query query = fieldQueryBuilder.build(getField(), keywords);

		setPreBooleanFilter(
			new TermFilter(Field.COMPANY_ID, String.valueOf(getCompanyId())),
			query);

		return query;
	}

	protected abstract FieldQueryBuilder createFieldQueryBuilder();

	protected Hits doSearch(String keywords) throws Exception {
		Query query = buildQuery(keywords);

		return search(createSearchContext(), query);
	}

	protected abstract String getField();

	protected void prepareSearch(
		IndexingTestHelper indexingTestHelper, String keywords) {

		indexingTestHelper.setQuery(buildQuery(keywords));
	}

	protected String[] transformFieldValues(String... values) {
		return null;
	}

	private void _assertCount(String keywords, int size) throws Exception {
		assertSearch(
			indexingTestHelper -> {
				prepareSearch(indexingTestHelper, keywords);

				indexingTestHelper.defineRequest(
					searchRequestBuilder -> searchRequestBuilder.size(
						size + 1));

				indexingTestHelper.search();

				indexingTestHelper.verify(
					hits -> DocumentsAssert.assertCount(
						keywords, hits.getDocs(), getField(), size));
			});
	}

}