/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.internal.storage.repository.git;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.linecorp.centraldogma.common.Revision;
import com.linecorp.centraldogma.common.RevisionNotFoundException;
import com.linecorp.centraldogma.server.internal.storage.StorageException;
import com.linecorp.centraldogma.server.internal.storage.repository.git.CommitUtil;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import javax.annotation.Nullable;
import org.eclipse.jgit.lib.AnyObjectId;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.revwalk.RevWalk;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class CommitIdDatabase
implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(CommitIdDatabase.class);
    private static final int RECORD_LEN = 24;
    private static final ThreadLocal<ByteBuffer> threadLocalBuffer = ThreadLocal.withInitial(() -> ByteBuffer.allocateDirect(24));
    private final Path path;
    private final FileChannel channel;
    private final boolean fsync;
    private volatile Revision headRevision;

    CommitIdDatabase(Repository repo) {
        this(repo.getDirectory(), repo.getConfig().getBoolean("core", "fsyncObjectFiles", false));
    }

    @VisibleForTesting
    CommitIdDatabase(File rootDir) {
        this(rootDir, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private CommitIdDatabase(File rootDir, boolean fsync) {
        this.path = new File(rootDir, "commit_ids.dat").toPath();
        try {
            this.channel = FileChannel.open(this.path, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
        }
        catch (IOException e) {
            throw new StorageException("failed to open a commit ID database: " + this.path, e);
        }
        this.fsync = fsync;
        boolean success = false;
        try {
            long size;
            try {
                size = this.channel.size();
            }
            catch (IOException e) {
                throw new StorageException("failed to get the file length: " + this.path, e);
            }
            if (size % 24L != 0L) {
                throw new StorageException("incorrect file length: " + this.path + " (" + size + " bytes)");
            }
            int numRecords = (int)(size / 24L);
            this.headRevision = numRecords > 0 ? new Revision(numRecords) : null;
            success = true;
        }
        finally {
            if (!success) {
                this.close();
            }
        }
    }

    @Nullable
    Revision headRevision() {
        return this.headRevision;
    }

    ObjectId get(Revision revision) {
        Revision headRevision = this.headRevision;
        Preconditions.checkState((headRevision != null ? 1 : 0) != 0, (String)"initial commit not available yet: %s", (Object)this.path);
        Preconditions.checkArgument((!revision.isRelative() ? 1 : 0) != 0, (String)"revision: %s (expected: an absolute revision)", (Object)revision);
        if (revision.major() > headRevision.major()) {
            throw new RevisionNotFoundException(revision);
        }
        ByteBuffer buf = threadLocalBuffer.get();
        buf.clear();
        long pos = (long)(revision.major() - 1) * 24L;
        try {
            do {
                int readBytes;
                if ((readBytes = this.channel.read(buf, pos)) < 0) {
                    throw new EOFException();
                }
                pos += (long)readBytes;
            } while (buf.hasRemaining());
        }
        catch (IOException e) {
            throw new StorageException("failed to read the commit ID database: " + this.path, e);
        }
        buf.flip();
        int actualRevision = buf.getInt();
        if (actualRevision != revision.major()) {
            throw new StorageException("incorrect revision number in the commit ID database: " + this.path + "(actual: " + actualRevision + ", expected: " + revision.major() + ')');
        }
        return new ObjectId(buf.getInt(), buf.getInt(), buf.getInt(), buf.getInt(), buf.getInt());
    }

    void put(Revision revision, ObjectId commitId) {
        this.put(revision, commitId, true);
    }

    private synchronized void put(Revision revision, ObjectId commitId, boolean safeMode) {
        if (safeMode) {
            Revision expected = this.headRevision == null ? Revision.INIT : this.headRevision.forward(1);
            Preconditions.checkState((boolean)revision.equals((Object)expected), (String)"incorrect revision: %s (expected: %s)", (Object)revision, (Object)expected);
        }
        ByteBuffer buf = threadLocalBuffer.get();
        buf.clear();
        buf.putInt(revision.major());
        commitId.copyRawTo(buf);
        buf.flip();
        long pos = (long)(revision.major() - 1) * 24L;
        try {
            do {
                pos += (long)this.channel.write(buf, pos);
            } while (buf.hasRemaining());
            if (safeMode && this.fsync) {
                this.channel.force(true);
            }
        }
        catch (IOException e) {
            throw new StorageException("failed to update the commit ID database: " + this.path, e);
        }
        if (safeMode || this.headRevision == null || this.headRevision.major() < revision.major()) {
            this.headRevision = revision;
        }
    }

    void rebuild(Repository gitRepo) {
        logger.warn("Rebuilding the commit ID database ..");
        try {
            this.channel.truncate(0L);
        }
        catch (IOException e) {
            throw new StorageException("failed to drop the commit ID database: " + this.path, e);
        }
        this.headRevision = null;
        try (RevWalk revWalk = new RevWalk(gitRepo);){
            Revision headRevision;
            ObjectId headCommitId = gitRepo.resolve("refs/heads/master");
            if (headCommitId == null) {
                throw new StorageException("failed to determine the HEAD: " + gitRepo.getDirectory());
            }
            RevCommit revCommit = revWalk.parseCommit((AnyObjectId)headCommitId);
            Revision previousRevision = headRevision = CommitUtil.extractRevision(revCommit.getFullMessage());
            block17: while (true) {
                RevCommit currentId;
                switch (revCommit.getParentCount()) {
                    case 0: {
                        break block17;
                    }
                    case 1: {
                        currentId = revCommit.getParent(0);
                        break;
                    }
                    default: {
                        throw new StorageException("found more than one parent: " + gitRepo.getDirectory());
                    }
                }
                revCommit = revWalk.parseCommit((AnyObjectId)currentId);
                Revision currentRevision = CommitUtil.extractRevision(revCommit.getFullMessage());
                Revision expectedRevision = previousRevision.backward(1);
                if (!currentRevision.equals((Object)expectedRevision)) {
                    throw new StorageException("mismatching revision: " + gitRepo.getDirectory() + " (actual: " + currentRevision.major() + ", expected: " + expectedRevision.major() + ')');
                }
                this.put(currentRevision, (ObjectId)currentId, false);
                previousRevision = currentRevision;
            }
            this.put(headRevision, headCommitId);
        }
        catch (Exception e) {
            throw new StorageException("failed to rebuild the commit ID database", e);
        }
        logger.info("Rebuilt the commit ID database.");
    }

    @Override
    public void close() {
        try {
            this.channel.close();
        }
        catch (IOException e) {
            logger.warn("Failed to close the commit ID database: {}", (Object)this.path, (Object)e);
        }
    }
}

