/*
 * Decompiled with CFR 0.152.
 */
package com.linecorp.centraldogma.server.internal.storage;

import com.linecorp.centraldogma.common.Author;
import com.linecorp.centraldogma.common.CentralDogmaException;
import com.linecorp.centraldogma.internal.Util;
import com.linecorp.centraldogma.server.internal.storage.StorageException;
import com.linecorp.centraldogma.server.internal.storage.StorageManager;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.regex.Pattern;
import javax.annotation.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class DirectoryBasedStorageManager<T>
implements StorageManager<T> {
    private static final Logger logger = LoggerFactory.getLogger(DirectoryBasedStorageManager.class);
    private static final Pattern CHILD_NAME = Pattern.compile("^[0-9A-Za-z](?:[-+_0-9A-Za-z.]*[0-9A-Za-z])?$");
    private static final String SUFFIX_REMOVED = ".removed";
    private final String childTypeName;
    private final Object[] childArgs;
    private final File rootDir;
    private final ConcurrentMap<String, T> children = new ConcurrentHashMap<String, T>();
    private final AtomicReference<Supplier<CentralDogmaException>> closed = new AtomicReference();

    protected DirectoryBasedStorageManager(File rootDir, Class<? extends T> childType, Object ... childArgs) {
        Objects.requireNonNull(rootDir, "rootDir");
        if (!rootDir.exists() && !rootDir.mkdirs()) {
            throw new StorageException("failed to create root directory at " + rootDir);
        }
        try {
            rootDir = rootDir.getCanonicalFile();
        }
        catch (IOException e) {
            throw new StorageException("failed to get the canonical path of: " + rootDir, e);
        }
        if (!rootDir.isDirectory()) {
            throw new StorageException("not a directory: " + rootDir);
        }
        this.rootDir = rootDir;
        this.childTypeName = Util.simpleTypeName(Objects.requireNonNull(childType, "childTypeName"), (boolean)true);
        this.childArgs = (Object[])Objects.requireNonNull(childArgs, "childArgs").clone();
        this.loadChildren();
    }

    protected Object childArg(int index) {
        return this.childArgs[index];
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void loadChildren() {
        boolean success = false;
        try {
            File[] childFiles = this.rootDir.listFiles();
            if (childFiles != null) {
                for (File f : childFiles) {
                    this.loadChild(f);
                }
            }
            success = true;
        }
        finally {
            if (!success) {
                this.close(() -> new CentralDogmaException("should never reach here"));
            }
        }
    }

    @Nullable
    private T loadChild(File f) {
        String name = f.getName();
        if (!DirectoryBasedStorageManager.isValidChildName(name)) {
            return null;
        }
        if (!f.isDirectory()) {
            return null;
        }
        if (new File(f + SUFFIX_REMOVED).exists()) {
            return null;
        }
        try {
            T child = this.openChild(f, this.childArgs);
            this.children.put(name, child);
            return child;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new StorageException("failed to open " + this.childTypeName + ": " + f, e);
        }
    }

    protected abstract T openChild(File var1, Object[] var2) throws Exception;

    protected abstract T createChild(File var1, Object[] var2, Author var3, long var4) throws Exception;

    private void closeChild(String name, T child, Supplier<CentralDogmaException> failureCauseSupplier) {
        this.closeChild(new File(this.rootDir, name), child, failureCauseSupplier);
    }

    protected void closeChild(File childDir, T child, Supplier<CentralDogmaException> failureCauseSupplier) {
    }

    protected abstract CentralDogmaException newStorageExistsException(String var1);

    protected abstract CentralDogmaException newStorageNotFoundException(String var1);

    @Override
    public void close(Supplier<CentralDogmaException> failureCauseSupplier) {
        Objects.requireNonNull(failureCauseSupplier, "failureCauseSupplier");
        if (!this.closed.compareAndSet(null, failureCauseSupplier)) {
            return;
        }
        for (Map.Entry e : this.children.entrySet()) {
            this.closeChild((String)e.getKey(), e.getValue(), failureCauseSupplier);
        }
    }

    @Override
    public boolean exists(String name) {
        this.ensureOpen();
        return this.children.containsKey(this.validateChildName(name));
    }

    @Override
    public T get(String name) {
        this.ensureOpen();
        Object child = this.children.get(this.validateChildName(name));
        if (child == null) {
            throw this.newStorageNotFoundException(name);
        }
        return (T)child;
    }

    @Override
    public T create(String name, long creationTimeMillis, Author author) {
        this.ensureOpen();
        Objects.requireNonNull(author, "author");
        this.validateChildName(name);
        AtomicBoolean created = new AtomicBoolean();
        Object child = this.children.computeIfAbsent(name, n -> {
            T c = this.create0(author, (String)n, creationTimeMillis);
            created.set(true);
            return c;
        });
        if (created.get()) {
            return (T)child;
        }
        throw this.newStorageExistsException(name);
    }

    private T create0(Author author, String name, long creationTimeMillis) {
        if (new File(this.rootDir, name + SUFFIX_REMOVED).exists()) {
            throw this.newStorageExistsException(name + " (removed)");
        }
        File f = new File(this.rootDir, name);
        boolean success = false;
        try {
            T newChild = this.createChild(f, this.childArgs, author, creationTimeMillis);
            success = true;
            T t = newChild;
            return t;
        }
        catch (RuntimeException e) {
            throw e;
        }
        catch (Exception e) {
            throw new StorageException("failed to create a new " + this.childTypeName + ": " + f, e);
        }
        finally {
            if (!success && f.exists()) {
                try {
                    Util.deleteFileTree((File)f);
                }
                catch (IOException e) {
                    logger.warn("Failed to delete a partially created project: {}", (Object)f);
                }
            }
        }
    }

    @Override
    public Map<String, T> list() {
        this.ensureOpen();
        int estimatedSize = this.children.size();
        Object[] names = this.children.keySet().toArray(new String[estimatedSize]);
        Arrays.sort(names);
        LinkedHashMap ret = new LinkedHashMap(estimatedSize);
        for (Object k : names) {
            Object v = this.children.get(k);
            if (v == null) continue;
            ret.put(k, v);
        }
        return Collections.unmodifiableMap(ret);
    }

    @Override
    public Set<String> listRemoved() {
        this.ensureOpen();
        LinkedHashSet<String> removed = new LinkedHashSet<String>();
        Object[] files = this.rootDir.listFiles();
        if (files == null) {
            return Collections.emptySet();
        }
        Arrays.sort(files);
        for (Object f : files) {
            String name;
            if (!((File)f).isDirectory() || !(name = ((File)f).getName()).endsWith(SUFFIX_REMOVED) || !DirectoryBasedStorageManager.isValidChildName(name = name.substring(0, name.length() - SUFFIX_REMOVED.length())) || this.children.containsKey(name)) continue;
            removed.add(name);
        }
        return Collections.unmodifiableSet(removed);
    }

    @Override
    public void remove(String name) {
        this.ensureOpen();
        Object child = this.children.remove(this.validateChildName(name));
        if (child == null) {
            throw this.newStorageNotFoundException(name);
        }
        this.closeChild(name, child, () -> this.newStorageNotFoundException(name));
        if (!new File(this.rootDir, name).renameTo(new File(this.rootDir, name + SUFFIX_REMOVED))) {
            throw new StorageException("failed to mark " + this.childTypeName + " as removed: " + name);
        }
    }

    @Override
    public T unremove(String name) {
        this.ensureOpen();
        this.validateChildName(name);
        File removed = new File(this.rootDir, name + SUFFIX_REMOVED);
        if (!removed.isDirectory()) {
            throw this.newStorageNotFoundException(name);
        }
        File unremoved = new File(this.rootDir, name);
        if (!removed.renameTo(unremoved)) {
            throw new StorageException("failed to mark " + this.childTypeName + " as unremoved: " + name);
        }
        T unremovedChild = this.loadChild(unremoved);
        if (unremovedChild == null) {
            throw this.newStorageNotFoundException(name);
        }
        return unremovedChild;
    }

    @Override
    public void ensureOpen() {
        if (this.closed.get() != null) {
            throw this.closed.get().get();
        }
    }

    private String validateChildName(String name) {
        if (!DirectoryBasedStorageManager.isValidChildName(Objects.requireNonNull(name, "name"))) {
            throw new IllegalArgumentException("invalid " + this.childTypeName + " name: " + name);
        }
        return name;
    }

    private static boolean isValidChildName(String name) {
        if (name == null) {
            return false;
        }
        if (!CHILD_NAME.matcher(name).matches()) {
            return false;
        }
        return !name.endsWith(SUFFIX_REMOVED);
    }

    public String toString() {
        return Util.simpleTypeName(this.getClass()) + '(' + this.rootDir + ')';
    }
}

