package com.liveperson.infra;

import android.os.Parcel;
import android.os.Parcelable;
import android.text.TextUtils;

import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.utils.Utils;

import java.util.ArrayList;
import java.util.List;



/**
 * Created by shiranr on 7/17/17.
 */

public class LPAuthenticationParams implements Parcelable {

    private static final String TAG = "LPAuthenticationParams";
    private LPAuthenticationType mType;
    private String mAuthKey;
    private String mHostAppJWT;
    private String mHostAppRedirectUri;
    private List<String> mCertificatePinningKeys;

    public enum LPAuthenticationType {SIGN_UP, UN_AUTH, AUTH }

    /**
     * Un-Authenticated
     */
    public LPAuthenticationParams() {
        //default should be sign up
        this(LPAuthenticationType.SIGN_UP);

    }

    public LPAuthenticationParams(LPAuthenticationType type) {
        mType = type;
    }

    protected LPAuthenticationParams(Parcel in) {
        mAuthKey = in.readString();
        mHostAppJWT = in.readString();
        mHostAppRedirectUri = in.readString();
        mCertificatePinningKeys = in.createStringArrayList();
        int i = in.readInt();
        if (i != -1){
            mType = LPAuthenticationType.values()[i];
        }
    }

    /**
     * Authentication with auth code only (v1)
     * Valid in LPAuthenticationType [AUTH]
     * @param authKey - Mandatory
     */
    public LPAuthenticationParams setAuthKey(String authKey) {
        mAuthKey = authKey;
        if (!TextUtils.isEmpty(mAuthKey)){
            mType = LPAuthenticationType.AUTH;
        }
        return this;
    }

    /**
     * Optional addition for auth code flow (v1)
     * Valid in LPAuthenticationType [AUTH]
     * @param hostAppRedirectUri
     * @return
     */
    public LPAuthenticationParams setHostAppRedirectUri(String hostAppRedirectUri) {
        mHostAppRedirectUri = hostAppRedirectUri;
        if (!TextUtils.isEmpty(mHostAppRedirectUri)){
            mType = LPAuthenticationType.AUTH;
        }
        return this;
    }

    /**
     * Authentication with auth jwt (v2)
     * Valid in LPAuthenticationType [AUTH]
     * @param hostAppJWT - Mandatory
     */
    public LPAuthenticationParams setHostAppJWT(String hostAppJWT) {
        this.mHostAppJWT = hostAppJWT;
        if (!TextUtils.isEmpty(mHostAppJWT)){
            mType = LPAuthenticationType.AUTH;
        }
        return this;
    }

     /**
     * Optional public keys, used for certificate pinning
     *
     * @param publicKey
     * @return
     */
    public LPAuthenticationParams addCertificatePinningKey(String publicKey) {
        if(TextUtils.isEmpty(publicKey)){
            LPMobileLog.d(TAG, "Certificate key can't be an empty string");
            return this;
        }
        if (mCertificatePinningKeys == null) {
            mCertificatePinningKeys = new ArrayList<>();
        }
        if(!publicKey.startsWith(Utils.SHA256)){
            publicKey = Utils.SHA256 + publicKey;
        }
        mCertificatePinningKeys.add(publicKey);

        return this;
    }

    public static final Creator<LPAuthenticationParams> CREATOR = new Creator<LPAuthenticationParams>() {
        @Override
        public LPAuthenticationParams createFromParcel(Parcel in) {
            return new LPAuthenticationParams(in);
        }

        @Override
        public LPAuthenticationParams[] newArray(int size) {
            return new LPAuthenticationParams[size];
        }
    };


    public String getHostAppJWT() {
        return mHostAppJWT;
    }

    public String getHostAppRedirectUri() {
        return mHostAppRedirectUri;
    }

    public String getAuthKey() {
        return mAuthKey;
    }

    public LPAuthenticationType getAuthType(){
        return mType;
    }

    public boolean isAuthenticated() {
        return mType == LPAuthenticationType.AUTH;//!TextUtils.isEmpty(mAuthKey) || !TextUtils.isEmpty(mHostAppJWT) ;
    }

    public List<String> getCertificatePinningKeys() {
        return mCertificatePinningKeys;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(mAuthKey);
        dest.writeString(mHostAppJWT);
        dest.writeString(mHostAppRedirectUri);
        dest.writeStringList(mCertificatePinningKeys);
        dest.writeInt(mType.ordinal());
    }
}
