package com.liveperson.infra.controller;

import android.content.Context;
import android.content.res.Resources;
import android.security.KeyPairGeneratorSpec;
import android.text.TextUtils;
import android.util.Base64;

import com.liveperson.infra.Clearable;
import com.liveperson.infra.Infra;
import com.liveperson.infra.R;
import com.liveperson.infra.configuration.Configuration;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.EncryptionVersion;
import com.liveperson.infra.utils.Utils;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.math.BigInteger;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.KeyStore;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Calendar;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.x500.X500Principal;

/**
 * Initialize this class to get access to the encryption key of the SQLite (SQLCipher).
 * The only public method is getDbEncryptionKey()
 * <p/>
 * Created by eyalv on 1/6/16.
 */
public class DBEncryptionKeyHelper implements Clearable {
    private static final String TAG = DBEncryptionKeyHelper.class.getSimpleName();

    private static final String DB_ENCRYPTION_KEY = "dbEncryptionKey";
    private static final String DB_ENCRYPTION_USES_KEYSTORE = "dbEncryptionUsesKeyStore";
    private static final String ANDROID_INFRA_DB_ENC_KEY = "androidInfraDbEncKey";
    private static final String DB_ENC_CN = "CN=DBKeyEncryptor, O=Liveperson";
    private static final String TRANSFORMATION_RSA = "RSA/ECB/PKCS1Padding";
    private static final String TRANSFORMATION_AES = "AES/CBC/PKCS5Padding";
    private static final String INITIALIZATION_VECTOR = "initializationVector";

    private SecretKey dbEncryptionKey;
    private KeyStore androidKeyStore = null;
    private SecureRandom secureRandom = new SecureRandom();
    IvParameterSpec ivSpec;


    /**
     * If we get an encryption key from the host app, we don't save it locally. If we don't get it from the host app, we use our own generated key.
     * On devices where AndroidKeyStore is supported, we save the generated key on shared prefs after encrypting the key with RSA KeyPair from the KeyStore. When not possible, the generated key is saved as is.
     *
     * @param externalEncryptionKey A DB encryption key supplied by the infra-hosting app.
     *                              If supplied on the first initialization after app installation, the same key needs to be supplied on any subsequent initialization.
     *                              It is then the responsibility of the host app to secure this key (for example: on server requiring login / on device requiring password, pin or fingerprint)
     *                              If null, a random secret key is generated and saved locally. If possible, using the AndroidKeyStore.
     */
    public DBEncryptionKeyHelper(final String externalEncryptionKey) {

        final String encryptionKey;

        // Fill the initialization vector for the CBC algorithm
        final String initializationVectorFromSharedPrefs = PreferenceManager.getInstance().getStringValue(INITIALIZATION_VECTOR, PreferenceManager.APP_LEVEL_PREFERENCES, null);
        byte[] iv;
        if (initializationVectorFromSharedPrefs == null) { // first time
            iv = new byte[16];
            secureRandom.nextBytes(iv);
            PreferenceManager.getInstance().setStringValue(INITIALIZATION_VECTOR, PreferenceManager.APP_LEVEL_PREFERENCES, Base64.encodeToString(iv, Base64.DEFAULT));
        } else {
            iv = Base64.decode(initializationVectorFromSharedPrefs, Base64.DEFAULT);
        }
        ivSpec = new IvParameterSpec(iv);


        try {
            if (isAndroidKeyStoreSupported()) {
                androidKeyStore = KeyStore.getInstance("AndroidKeyStore");
                androidKeyStore.load(null, null);
            }
        } catch (Exception e) {
            LPMobileLog.e(TAG, e.getMessage()); // TODO: 1/10/16
        }

        if (externalEncryptionKey == null) {
            // We didn't get the encryptionKey from the host app

            // Try to get the encrypted encryption key from shared prefs
            final String encryptedEncryptionKeyFromSharedPrefs = PreferenceManager.getInstance().getStringValue(DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, null);

			// Try to get the privateKey from the keyStore. This is done here only for testing whether the keystore already holds the key.
			// In case of AutomaticRestore is enabled on the device, the restored data does not include
			// the keys from the keyStore, so if we don't have it, we behave as we need to recreate it.
			KeyStore.PrivateKeyEntry privateKeyEntryForValidation;
			try {
				privateKeyEntryForValidation = (KeyStore.PrivateKeyEntry) androidKeyStore.getEntry(ANDROID_INFRA_DB_ENC_KEY, null);
			} catch (Exception e) {
				privateKeyEntryForValidation = null;
			}

			// If either the key from
			if ((encryptedEncryptionKeyFromSharedPrefs == null) || (privateKeyEntryForValidation == null)) {
                // First time - doesn't exist in shared prefs yet
                encryptionKey = generateAesEncryptionKey();
                encryptAndSaveKey(encryptionKey);
            } else {
                // Got the key from shared prefs
                boolean isKeyEncryptedUsingKeyStore = PreferenceManager.getInstance().getBooleanValue(DB_ENCRYPTION_USES_KEYSTORE, PreferenceManager.APP_LEVEL_PREFERENCES, false);
                if (isKeyEncryptedUsingKeyStore) {
                    encryptionKey = decryptKey(encryptedEncryptionKeyFromSharedPrefs);
                } else {
                    encryptionKey = encryptedEncryptionKeyFromSharedPrefs; // The key is not encrypted. Use it as is.

                    if (isAndroidKeyStoreSupported()) {
                        // The device now supports AndroidKeyStore. Probably after OS upgrade. Encrypt & save the encrypted key instead of the unencrypted one.
                        encryptAndSaveKey(encryptionKey);
                    }
                }
            }
        } else {
            // We got the encryptionKey from the host app
            encryptionKey = externalEncryptionKey;
        }

        setDbEncryptionKey(encryptionKey);
    }


    private String decryptKey(String encryptedKey) {
        String decryptedKey = null;

        try {
            KeyStore.PrivateKeyEntry privateKeyEntry = (KeyStore.PrivateKeyEntry) androidKeyStore.getEntry(ANDROID_INFRA_DB_ENC_KEY, null);
            Cipher output = Cipher.getInstance(TRANSFORMATION_RSA);
            output.init(Cipher.DECRYPT_MODE, privateKeyEntry.getPrivateKey());

            CipherInputStream cipherInputStream = new CipherInputStream(new ByteArrayInputStream(Base64.decode(encryptedKey, Base64.DEFAULT)), output);
            ArrayList<Byte> values = new ArrayList<>();
            int nextByte;
            while ((nextByte = cipherInputStream.read()) != -1) {
                values.add((byte) nextByte);
            }

            byte[] bytes = new byte[values.size()];
            for (int i = 0; i < bytes.length; i++) {
                bytes[i] = values.get(i);
            }

            decryptedKey = new String(bytes, 0, bytes.length, "UTF-8");
        } catch (Exception e) {
            LPMobileLog.e(TAG, e); // TODO: 1/13/16
        }
        //LPMobileLog.d(TAG, "DB Key decrypt. Before: " + encryptedKey + " after: " + decryptedKey);
        return decryptedKey;
    }


    private String encryptAndSaveKey(final String decryptedKey) {
        String encryptedKey;
        boolean usesAndroidKeyStore = isAndroidKeyStoreSupported();
        if (usesAndroidKeyStore) {
            generateKeyPairInStoreIfNotExists(ANDROID_INFRA_DB_ENC_KEY);
            try {
                KeyStore.PrivateKeyEntry publicKeyEntry = (KeyStore.PrivateKeyEntry) androidKeyStore.getEntry(ANDROID_INFRA_DB_ENC_KEY, null);

                Cipher input = Cipher.getInstance(TRANSFORMATION_RSA);
                input.init(Cipher.ENCRYPT_MODE, publicKeyEntry.getCertificate().getPublicKey());

                ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
                CipherOutputStream cipherOutputStream = new CipherOutputStream(outputStream, input);
                cipherOutputStream.write(decryptedKey.getBytes("UTF-8"));
                cipherOutputStream.close();

                byte[] vals = outputStream.toByteArray();
                encryptedKey = Base64.encodeToString(vals, Base64.DEFAULT);
            } catch (Exception e) {
                LPMobileLog.e(TAG, e); // TODO: 1/13/16
                encryptedKey = decryptedKey;
                usesAndroidKeyStore = false;
            }
        } else {
            encryptedKey = decryptedKey;
        }

        // Save encrypted key
        PreferenceManager.getInstance().setStringValue(DB_ENCRYPTION_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, encryptedKey); // put the encrypted key on shared prefs for the next time
        PreferenceManager.getInstance().setBooleanValue(DB_ENCRYPTION_USES_KEYSTORE, PreferenceManager.APP_LEVEL_PREFERENCES, usesAndroidKeyStore); // Mark as encrypted using keystore

        return encryptedKey;
    }


    private String generateAesEncryptionKey() {
        SecretKey secretKey;

        try {
            KeyGenerator keyGen = KeyGenerator.getInstance("AES");
            keyGen.init(256); // TODO: 2/8/16 use 192 or 128 if not available
            secretKey = keyGen.generateKey();
        } catch (NoSuchAlgorithmException e) {
            LPMobileLog.e(TAG, e);
            byte[] randomBytes = new byte[32];
            secureRandom.nextBytes(randomBytes);
            return Base64.encodeToString(randomBytes, Base64.DEFAULT); // TODO: 1/7/16 Check how to better handle this exception.
        }

        String encodedKey = Base64.encodeToString(secretKey.getEncoded(), Base64.DEFAULT); // TODO: 1/11/16 check if this is right
        return encodedKey;
    }


    /**
     * This method verifies that the key pair exists in the Android key store. If not, it creates the pair.
     * On Android versions below 4.3, where AndroidKeyStore is not available, the method returns null
     *
     * @return the generated/existing key pair or null if Android version is below 4.3
     */
    private KeyPair generateKeyPairInStoreIfNotExists(final String alias) {
        KeyPair keyPair = null;
        if (isAndroidKeyStoreSupported()) {
            try {
                // Create new key if needed
                if (!androidKeyStore.containsAlias(alias)) {
                    Calendar start = Calendar.getInstance();
                    Calendar end = Calendar.getInstance();
                    end.add(Calendar.YEAR, 120);
                    KeyPairGeneratorSpec spec = new KeyPairGeneratorSpec.Builder(Infra.instance.getApplicationContext())
                            .setAlias(alias)
                            .setSubject(new X500Principal(DB_ENC_CN))
                            .setSerialNumber(BigInteger.ONE)
                            .setStartDate(start.getTime())
                            .setEndDate(end.getTime())
                            .build(); // TODO: 1/10/16 Use KeyGenParameterSpec instead
                    KeyPairGenerator generator = KeyPairGenerator.getInstance("RSA", "AndroidKeyStore");
                    generator.initialize(spec);

                    keyPair = generator.generateKeyPair();
                }
            } catch (Exception e) {
                LPMobileLog.e(TAG, e); // TODO: 1/11/16
            }
        }
        return keyPair;
    }

    private boolean isAndroidKeyStoreSupported() {
        return android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.JELLY_BEAN_MR2;
    }

    private void setDbEncryptionKey(final String key) {
        byte[] decodedKey = Base64.decode(key, Base64.DEFAULT);
        decodedKey[0]++;
        SecretKey secretKey = new SecretKeySpec(decodedKey, "AES");
        dbEncryptionKey = secretKey;
    }

    private SecretKey getDbEncryptionKey() {
        return dbEncryptionKey;
    }

    String encrypt(final String plainText) {
        if(!TextUtils.isEmpty(plainText)) {
            try {
                Cipher cipher = Cipher.getInstance(TRANSFORMATION_AES);
                cipher.init(Cipher.ENCRYPT_MODE, getDbEncryptionKey(), ivSpec);
                byte[] encryptedBytes = cipher.doFinal(plainText.getBytes("UTF-8"));
                String cipherText = Base64.encodeToString(encryptedBytes, Base64.DEFAULT);
                return cipherText;
            } catch (Exception e) {
                LPMobileLog.e(TAG, e);
            }
        }
        return plainText; // TODO: 2/15/16
    }


    String decrypt(final String cipherText) {
        String plainText = "";
        if(!TextUtils.isEmpty(cipherText)) {
            try {
                Cipher cipher = Cipher.getInstance(TRANSFORMATION_AES);
                cipher.init(Cipher.DECRYPT_MODE, getDbEncryptionKey(), ivSpec);

                byte[] data = Base64.decode(cipherText, Base64.DEFAULT);
                byte[] decryptedBytes = cipher.doFinal(data);

                plainText = new String(decryptedBytes, "UTF-8");
            } catch (BadPaddingException badPaddingException) {
                // Bug LE-91533 occurred
                LPMobileLog.e(TAG, "Caught a bad padding exception!", badPaddingException);
                LPMobileLog.d(TAG, LPMobileLog.FlowTags.DECRYPTION, "Using fallback after BadPaddingException");
                try {
                    Cipher cipher = Cipher.getInstance(TRANSFORMATION_AES);
                    cipher.init(Cipher.DECRYPT_MODE, getDbEncryptionKey(), ivSpec);

                    // Solution from: https://stackoverflow.com/questions/4580982/javax-crypto-badpaddingexception
                    byte[] data = Utils.hexStringToByteArray(cipherText);
                    byte[] decryptedBytes = cipher.doFinal(data);

                    plainText = new String(decryptedBytes, "UTF-8");
                    LPMobileLog.d(TAG, LPMobileLog.FlowTags.DECRYPTION, "BadPaddingException fallback worked!");
                } catch (Exception e) {
                    LPMobileLog.e(TAG, e);
                }
            } catch (Exception e) {
                LPMobileLog.e(TAG, e);
            }
        }

        return plainText; // TODO: 2/8/16
    }

    public static EncryptionVersion getAppEncryptionVersion(Context appContext) {
        int encryptionVersion;
        try {
            encryptionVersion = Configuration.getInteger(R.integer.encryptionVersion);
            return EncryptionVersion.fromInt(encryptionVersion);
        } catch (Resources.NotFoundException e) {
            LPMobileLog.e(TAG, e);
            return EncryptionVersion.VERSION_1;
        }
    }


    @Override
    public void clear() {
        if (androidKeyStore != null) {
            try {
                if (androidKeyStore.containsAlias(ANDROID_INFRA_DB_ENC_KEY)){
                    androidKeyStore.deleteEntry(ANDROID_INFRA_DB_ENC_KEY);
                }
            } catch (Exception kse) {
                LPMobileLog.d(TAG, "exception deleting key store entry: " + kse.getMessage());
            }
        }
    }

}
