package com.liveperson.infra.log;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.logreporter.loggos.Loggos;
import com.liveperson.infra.log.logreporter.loggos.LoggosMessage;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

/**
 * Log helper.
 * Should be in use only by LP SDK, not by the host app or others.
 *
 * @author ofira
 */
public class LPMobileLog {

	private static final String TAG = "Liveperson";
    private static boolean DEBUG = true; //TODO: use BuildConfig.DEBUG
    private static boolean isLoggosEnabled = true;
	private static final String LOGGOS_CONTEXT = "AndroidSdk";
	private static final String INFO = "INFO";
	private static final String WARN = "WARN";
	private static final String ERROR = "ERROR";
	private static ILogger logger = new DefaultAndroidLogger();

    public static void setLogger(ILogger logger) {
        LPMobileLog.logger = logger;
    }

    /** Set If we want to write logs to Loggos.
     * This is false If we are running in test environment.
     *
     *  @param enabled
     **/
    public static void setLoggosEnabled(boolean enabled) {
        isLoggosEnabled = enabled;
    }

    public static void setDebugMode(boolean debugMode) {
        LPMobileLog.DEBUG = debugMode;
    }

    /**
     * Log a piece of debug information.
     *
     * @param logMessage
     */
    public static void d(String classTAG, String logMessage) {
        if (DEBUG) {
            logger.d(TAG + " **" + classTAG + "**", logMessage);
        }
    }

    /**
     * Log a piece of debug information while adding a flow tag.
     *
     * @param string
     */
    public static void d(String classTAG, String flowTag, String string) {
        d(classTAG, "--" + flowTag + "-- " + string);
    }

    /**
     * Log a piece of information while adding a flow tag.
     *
     * @param string
     */
    public static void i(String classTAG, String flowTag, String string) {
        i(classTAG, "--" + flowTag + "-- " + string);
    }

    /**
     * Log a piece of information.
     *
     * @param string
     */
    public static void i(String classTAG, String string) {
        if (DEBUG) {
            logger.i(TAG + " **" + classTAG + "**", string);
		}
        addToLoggos(classTAG, string, INFO);
    }

    /**
     * Log a piece of debug information.
     *
     * @param string
     */
    public static void w(String classTAG, String string) {
        if (DEBUG) {
            logger.w(TAG + " **" + classTAG + "**", string);
        }
        addToLoggos(classTAG, string, WARN);
    }

    public static void w(String classTAG, Throwable t) {
        if (DEBUG) {
            logger.w(TAG + " **" + classTAG + "**", WARN + " " + t.getMessage());
		}
        addToLoggos(classTAG, getStackTrace(t), WARN);
    }

    /**
     * Log an error.
     *
     * @param errorMessage
     */
    public static void e(String classTAG, String errorMessage) {
        logger.e(TAG + " **" + classTAG + "**", errorMessage);
        sendToLoggosNow(classTAG, errorMessage, ERROR);
    }

    /**
     * Logs an exception and prints its stack trace.
     *
     * @param error
     */
    public static void e(String classTAG, Throwable error) {
        e(classTAG, null, error);
    }

    /**
     * Log an error.
     *
     * @param classTAG
     * @param flowTag
     * @param errorMessage
     */
    public static void e(String classTAG, String flowTag, String errorMessage) {
        e(classTAG, "--" + flowTag + "-- " + errorMessage);
    }


    /**
     * Logs an error with its exception + stack trace.
     * @param classTAG
     * @param errorMessage
     * @param error
     */
    public static void e(String classTAG, String errorMessage, Throwable error) {
        if (errorMessage != null) {
            logger.e(TAG + " **" + classTAG + "**", errorMessage);
        }
        error.printStackTrace();

        sendToLoggosNow(classTAG, getStackTrace(error), ERROR);
    }

    private static String getStackTrace(Throwable aThrowable) {
        if (aThrowable == null) return "";
		final Writer result = new StringWriter();
		final PrintWriter printWriter = new PrintWriter(result);
		aThrowable.printStackTrace(printWriter);
		return result.toString();
	}

	private static void addToLoggos(String classTAG, String message, String level) {
        if(isLoggosEnabled) {
            LoggosMessage loggosMessage = new LoggosMessage(level, LOGGOS_CONTEXT, message, classTAG);
            Loggos loggos = Infra.instance.getLoggos();
            if (loggos != null) {
                loggos.addMessage(loggosMessage);
            }
        }
	}

	private static void sendToLoggosNow(String classTAG, String message, String level) {
        if(isLoggosEnabled) {
            LoggosMessage loggosMessage = new LoggosMessage(level, LOGGOS_CONTEXT, message, classTAG);
            Loggos loggos = Infra.instance.getLoggos();
            if (loggos != null) {
                loggos.sendMessageImmediately(loggosMessage);
            }
        }
	}

    public class FlowTags {
        public static final String DIALOGS = "MULTI_DIALOG_FLOW";
        public static final String LOGIN = "LOGIN_FLOW";
        public static final String DECRYPTION = "DECRYPTION";
    }
}