package com.liveperson.infra.log.logreporter.loggos;

import android.text.TextUtils;

import com.liveperson.infra.InternetConnectionService;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.log.logreporter.loggos.logsender.HttpLogSender;
import com.liveperson.infra.log.logreporter.loggos.logsender.LogSender;
import com.liveperson.infra.log.logreporter.loggos.logsender.LogSenderCallback;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.utils.LocalBroadcast;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.net.ssl.SSLPeerUnverifiedException;

/**
 * Created by nirni on 3/16/16.
 */
public class Loggos implements LogSenderCallback<Exception>{

	public static final String TAG = Loggos.class.getSimpleName();

	public static final String PREFS_KEY_LOGGOS_DOMAIN = "prefs_key_loggos_domain";
	public static final String PREFS_KEY_LOGGOS_TARGET_ID = "prefs_key_loggos_targetid";
	private static final String DEFAULT_BRAND_ID_FOR_PREFS = "default_brand";
	private static final String PREFS_KEY_LAST_TIME_ERROR_SENT_PREFS = "prefs_key_last_time_error_sent";

	public static final String CERTIFICATE_ERROR_ACTION = "certificate_error_action";

	private static final long NUM_OF_MILLISECONDS_BETWEEN_ERRORS = 60*60*1000;
	private static final int MAX_QUEUE_SIZE = 100;

	private LogSender mLogSender = new HttpLogSender();
	private String mTargetId;
	private String mDomain;
	private List<String> mCertificates;
	private final Object mQueueLock = new Object();

	private long mLastTimeErrorSent;
	private LinkedList<LoggosMessage> mMessageQueue = new LinkedList<>();

	public Loggos() {
		validateDomainAndBrand();
	}

	public void init(String targetId, String domain, List<String> certificates) {
		mTargetId = targetId;
		mDomain = domain;
		mCertificates = certificates;
		mLogSender.setCallback(new LogSenderCallback() {
			@Override
			public void onSuccess() {

			}

			@Override
			public void onError(List list, Throwable exception) {
				if(exception instanceof SSLPeerUnverifiedException){
					String intentAction = CERTIFICATE_ERROR_ACTION;
					LocalBroadcast.sendBroadcast(intentAction);
				}
			}
		});

		// Store in preferences
		LPMobileLog.d(TAG, "init: save domain: " + mDomain + ", targetId: " + mTargetId + " to prefs");
		PreferenceManager.getInstance().setStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, mDomain);
		PreferenceManager.getInstance().setStringValue(PREFS_KEY_LOGGOS_TARGET_ID, DEFAULT_BRAND_ID_FOR_PREFS, mTargetId);
	}

	public void shutdown() {

		sendAllQueue();
	}

	/**
	 * Send the message immediately (including all pending messages)
	 *
	 * @param message
	 */
	public void sendMessageImmediately(LoggosMessage message) {

		LPMobileLog.d(TAG, "sendMessageImmediately: Sending message");

		if (message == null) {
			LPMobileLog.d(TAG, "sendMessage: log message is null. Did not send to Loggos");
			return;
		}

		// If we're not initialized yet or there is no network we just add the message to the queue
		if ((mDomain == null) || (mTargetId == null) || !isNetworkAvailable()) {
			LPMobileLog.d(TAG, "sendMessage: log sender is not initialized yet. Store the message for future sending");

			addMessageToQueue(message);
			return;
		}

		// Set target id if missing
		if (TextUtils.isEmpty(message.getAccountId())) {
			message.setAccountId(mTargetId);
		}

		// Add the message to the queue and send all queue
		addMessageToQueue(message);

		sendAllQueue();
	}

	/**
	 * Add a message to the internal queue and send all queue if passed threshold
	 * @param message
	 */
	public void addMessage(LoggosMessage message) {

		if (message == null) {
			LPMobileLog.d(TAG, "addMessage: log message is null. Did not send to Loggos");
			return;
		}

		if (TextUtils.isEmpty(message.getAccountId()) && (mTargetId != null)) {
			message.setAccountId(mTargetId);
		}

		addMessageToQueue(message);

	}

	/**
	 *
	 * @param message
	 */
	private void addMessageToQueue(LoggosMessage message) {

		synchronized (mQueueLock) {

			if (mMessageQueue.size() >= MAX_QUEUE_SIZE) {

				mMessageQueue.removeLast();
			}

			mMessageQueue.addFirst(message);
		}
	}

	/**
	 * Send all messages from queue
	 */
	private void sendAllQueue(){

		if( isNetworkAvailable() && isShouldSendError()) {
			List<LoggosMessage> listToSend = new ArrayList<>();

			LPMobileLog.d(TAG, "sendAllQueue: Start: " + System.currentTimeMillis());
			synchronized (mQueueLock) {
				LPMobileLog.d(TAG, "sendAllQueue: Copy message list and clear " + mMessageQueue.size() + " messages");
				listToSend.addAll(mMessageQueue);
				mMessageQueue.clear();

				LPMobileLog.d(TAG, "sendAllQueue: Sending all repository, " + listToSend.size() + " messages");
				mLogSender.sendBulk(mDomain, listToSend, mCertificates);

				// Store the time
				mLastTimeErrorSent = System.currentTimeMillis();
				PreferenceManager.getInstance().setLongValue(PREFS_KEY_LAST_TIME_ERROR_SENT_PREFS, DEFAULT_BRAND_ID_FOR_PREFS, mLastTimeErrorSent);
			}

			LPMobileLog.d(TAG, "sendAllQueue: End: " + System.currentTimeMillis());

		}
		else{
			LPMobileLog.d(TAG, "sendAllQueue: Network is not available or didn't pass enough time for next sending. Didn't send messages");
		}
	}

	/**
	 * Validate we have domain and targetId and if required get from preferences
	 */
	private void validateDomainAndBrand() {
		if (mDomain == null) {
			// Get from preferences
			mDomain = PreferenceManager.getInstance().getStringValue(PREFS_KEY_LOGGOS_DOMAIN, DEFAULT_BRAND_ID_FOR_PREFS, null);
			LPMobileLog.d(TAG, "validateDomainAndBrand: Get domain from preferences: " + mDomain);
		}

		if (mTargetId == null) {
			// Get from preferences
			mTargetId = PreferenceManager.getInstance().getStringValue(PREFS_KEY_LOGGOS_TARGET_ID, DEFAULT_BRAND_ID_FOR_PREFS, null);
			LPMobileLog.d(TAG, "validateDomainAndBrand: Get targetId from preferences: " + mTargetId);
		}
	}

	private boolean isNetworkAvailable(){

		return InternetConnectionService.isNetworkAvailable();
	}

	public boolean isShouldSendError() {

		mLastTimeErrorSent = PreferenceManager.getInstance().getLongValue(PREFS_KEY_LAST_TIME_ERROR_SENT_PREFS, DEFAULT_BRAND_ID_FOR_PREFS, 0);
		boolean needToSend = (System.currentTimeMillis() - mLastTimeErrorSent) > NUM_OF_MILLISECONDS_BETWEEN_ERRORS;
		if (!needToSend) {
			LPMobileLog.d(TAG, "Time from last error is less than " + NUM_OF_MILLISECONDS_BETWEEN_ERRORS + " millis. No need to send to loggos");
		}

		return needToSend;
	}

	public String getTargetId() {
		return mTargetId;
	}

	public void setTargetId(String targetId) {
		mTargetId = targetId;
	}

	public String getDomain() {
		return mDomain;
	}

	public void setDomain(String domain) {
		mDomain = domain;
	}


	@Override
	public void onSuccess() {

		LPMobileLog.d(TAG, "onSuccess: Messages sent successfully");
	}

	@Override
	public void onError(List<LoggosMessage> messages, Exception e) {

		LPMobileLog.d(TAG, "onError: Error sending messages. Add them back to top of stack (exception: " + e + ")");
		synchronized (mQueueLock) {
			mMessageQueue.addAll(0, messages);
		}
	}
}
