package com.liveperson.infra.handler;


import android.app.Notification;
import android.app.Notification.Builder;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.support.annotation.RequiresApi;

import com.liveperson.infra.R;

/**
 * Created by Ilya Gazman on 11/26/2015.
 */
public class NotificationHandler {

    private static final int NOTIFICATION_ID = 1212322;

    private final String mTitle;
    private final String mMessage;
    private final Context mContext;

    private int mIconResourceId;
    private PendingIntent mPendingIntent;

    public NotificationHandler(Context context, String title, String message) {
        mContext = context;
        mTitle = title;
        mMessage = message;
    }

    /**
     * Set the icon resource for the notification
     *
     * @param iconResourceId Resource id of icon.
     */
    public NotificationHandler setIconResourceId(int iconResourceId) {
        mIconResourceId = iconResourceId;
        return this;
    }

    /**
     * Set the pending intent for the notification
     */
    public NotificationHandler setPendingIntent(PendingIntent pendingIntent) {
        mPendingIntent = pendingIntent;
        return this;
    }

    /**
     * Hide notification
     */
    public static void hideNotification(Context context, String brandId) {
        ((NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE)).cancel(brandId, NOTIFICATION_ID);
    }

    /**
     * Show notification
     */
    public void showNotification() {
        Notification notification = createPushNotificationBuilder().build();
        getNotificationManager().notify(NOTIFICATION_ID, notification);
    }

    private Builder createPushNotificationBuilder() {
        String channelId = mContext.getString(R.string.push_notification_channel_id);
        String channelName = mContext.getString(R.string.push_notification_channel_name);

        Builder builder = createNotificationBuilder(channelId, channelName);
        builder.setContentTitle(mTitle)
                .setAutoCancel(true)
                .setDefaults(Notification.DEFAULT_SOUND | Notification.DEFAULT_LIGHTS)
                .setSmallIcon(mIconResourceId)
                .setContentText(mMessage);

        if (Build.VERSION.SDK_INT >= 21) {
            builder.
                    setCategory(Notification.CATEGORY_MESSAGE).
                    setPriority(Notification.PRIORITY_HIGH);
        }

        return builder;
    }

    public Builder createForegroundServiceNotificationBuilder() {
        String channelId = mContext.getString(R.string.foreground_service_notification_channel_id);
        String channelName = mContext.getString(R.string.foreground_service_notification_channel_name);

        Builder builder = createNotificationBuilder(channelId, channelName);
        builder = builder
                .setContentTitle(mTitle)
                .setSmallIcon(mIconResourceId)
                .setLargeIcon(BitmapFactory.decodeResource(mContext.getResources(), mIconResourceId))
                .setContentIntent(mPendingIntent)
                .setProgress(0, 0, true);

        return builder;
    }

    private Notification.Builder createNotificationBuilder(String channelId, String channelName) {
        Notification.Builder builder;
        if (Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.O) {
            builder = new Notification.Builder(mContext);
        } else {
            createNotificationChannel(channelId, channelName);
            builder = new Notification.Builder(mContext, channelId);
        }

        return builder;
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private void createNotificationChannel(String channelId, String channelName){
        NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, NotificationManager.IMPORTANCE_DEFAULT);
        getNotificationManager().createNotificationChannel(notificationChannel);
    }

    private NotificationManager getNotificationManager() {
        return (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);
    }
}
