package com.liveperson.infra.utils;

import android.content.ContentResolver;
import android.content.Context;
import android.content.res.TypedArray;
import android.database.Cursor;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;
import android.support.annotation.NonNull;
import android.support.media.ExifInterface;
import android.support.v4.content.FileProvider;
import android.text.TextUtils;
import android.util.Base64;
import android.util.DisplayMetrics;
import android.view.WindowManager;

import com.liveperson.infra.Infra;
import com.liveperson.infra.log.LPLog;
import com.squareup.picasso.Picasso;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.UUID;

import static com.liveperson.infra.errors.ErrorCode.ERR_00000018;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000019;
import static com.liveperson.infra.errors.ErrorCode.ERR_0000001A;
import static com.liveperson.infra.utils.Utils.getResources;

/**
 * Created by nirni on 7/3/16.
 */
public class ImageUtils {

	private static final String TAG = "ImageUtils";

	public static final String PNG = "png";
	public static final String JPG = "jpg";
	private static final String SAMSUNG = "samsung";
	public static final String FILE_MIME_TYPE_GIF = "image/gif";
	public static final String FILE_MIME_TYPE_JPG = "image/jpeg";

	private static final String IMAGES_FOLDER = "images/";
	private static final String FULL_IMAGE_FOLDER = "full/";
	private static final String PREVIEW_IMAGE_FOLDER = "preview/";

	public enum ImageFolderType {PREVIEW, FULL}

	public static String bitmapToBase64(byte[] bitmapByteArray) {

		LPLog.INSTANCE.d(TAG, "bitmapToBase64: Bitmap size: " + bitmapByteArray.length / 1000 + " kb");
		return Base64.encodeToString(bitmapByteArray, Base64.DEFAULT);
	}

	/**
	 * Convert base64 string to an image byte array. Since the base64 received is with a prefix (data:image/jpeg;base64,)
	 * and the decoder does not decode the base64 correctly with it, we need to remove it before decoding
	 */
	private static byte[] base64ToByteArray(String base64) {

		if (base64 != null) {
			LPLog.INSTANCE.d(TAG, "base64ToByteArray: converting base64 to byte array");

			// Remove any existing base64 prefix (e.g. data:image/jpeg;base64,)
			base64 = base64.replaceFirst("data.*base64,", "");

			return Base64.decode(base64, Base64.DEFAULT);
		}

		return null;
	}

	public static String bitmapToBase64(Bitmap bitmap) {

		ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
		bitmap.compress(Bitmap.CompressFormat.PNG, 100, byteArrayOutputStream);
		byte[] byteArray = byteArrayOutputStream.toByteArray();

		LPLog.INSTANCE.d(TAG, "bitmapToBase64: Bitmap size: " + byteArray.length / 1000 + " kb");
		return Base64.encodeToString(byteArray, Base64.DEFAULT);
	}

	public static Bitmap createResizedBitmap(Uri imageUri, int reqLongerSide, int orientation, boolean fromCamera) throws IOException {

		Bitmap bitmap = Picasso.get().load(imageUri).resize(reqLongerSide, reqLongerSide).onlyScaleDown().centerInside().get();

		// When picasso resize the img, The Exif not been saved
		// Create a new bitmap with the original orientation (from the exif)
		Matrix matrix = new Matrix();
		matrix.postRotate(decodeExifOrientation(orientation, fromCamera));

		return Bitmap.createBitmap(bitmap, 0, 0, bitmap.getWidth(), bitmap.getHeight(), matrix, true);
	}

	/**
	 * createBitmapFromURL - generate a bitmap image from a url with specific resize
	 */
	public static Bitmap createBitmapFromURL(Context context, String imageUrl, int width, int height) {
		if (TextUtils.isEmpty(imageUrl)) {
			return null;
		}
		try {
			return PicassoUtils.get(context).load(imageUrl).resize(Math.max(width, 250), Math.max(height, 250)).onlyScaleDown().get();

		} catch (Exception e) {
			try {
				return PicassoUtils.get(context).load(imageUrl).get();
			} catch (Exception e2) {
				return null;
			}
		}
	}

	@NonNull
	public static Bitmap getBitmap(String imagePath) {
		final BitmapFactory.Options options = new BitmapFactory.Options();
		Bitmap bitmap = BitmapFactory.decodeFile(imagePath, options);
		bitmap = Bitmap.createScaledBitmap(bitmap, bitmap.getWidth(), bitmap.getHeight(), true);

		return bitmap;
	}

	public static int getImageRotation(String imagePath, boolean fromCamera) {

		try {
			ExifInterface exif = new ExifInterface(imagePath);
			int orientation = exif.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
			return decodeExifOrientation(orientation, fromCamera);
		} catch (IOException e) {
			LPLog.INSTANCE.i(TAG, "getImageRotation: cannot get exif information on image. Return rotation 0");
			return 0;
		}

	}

	/**
	 * This returns the path of the image given by the Uri. If the Uri is with 'content' scheme it gets
	 * the path from the content resolver. If the scheme is 'file' it simply gets the path from the Uri.
	 * Otherwise this method returns null.
	 *
	 * @return the image path, or null if cannot extract it
	 */
	public static String getImagePath(Context context, Uri selectedImageUri) {

		String path = null;
		String scheme = selectedImageUri.getScheme();

		// If scheme is 'content'
		if (scheme != null && scheme.equals(ContentResolver.SCHEME_CONTENT)) {
			String authority = selectedImageUri.getAuthority();

			// The prefix here is the same as defined in the provider in the manifest
			if (authority != null && authority.equals(Infra.getFileProviderAuthorityPrefix() + context.getApplicationInfo().packageName)) {
				//file saved in local app folder, most probably when user took a photo with camera
				path = new File(context.getFilesDir(), selectedImageUri.getPath()).getAbsolutePath();
			} else {
				String[] filePathColumn = {MediaStore.Images.Media.DATA};
				Cursor cursor = context.getContentResolver().query(selectedImageUri, filePathColumn, null, null, null);
				if (cursor != null) {

					cursor.moveToFirst();
					int columnIndex = cursor.getColumnIndex(filePathColumn[0]);
					path = cursor.getString(columnIndex);
					cursor.close();
					if (TextUtils.isEmpty(path)) {
						path = selectedImageUri.toString();
					}
				}
			}

		}
		// If scheme is 'file'
		else if (scheme != null && scheme.equals(ContentResolver.SCHEME_FILE)) {
			LPLog.INSTANCE.d(TAG, "getImagePath: Uri scheme is file. We get the path of the file");
			path = selectedImageUri.getPath();
		}
		// Unknown scheme
		else {
			LPLog.INSTANCE.e(TAG, ERR_00000018, "getImagePath: Unknown scheme: " + scheme + ". Cannot get image path");
		}

		LPLog.INSTANCE.d(TAG, "getImagePath: path = " + path);
		return path;
	}

	/**
	 * This method is used to compress and get a byte array for the given bitmap
	 */
	public static byte[] getOutputStreamFromBitmap(Bitmap bitmapImage, int compression, String imageTypeExtension) {

		byte[] byteArray = null;

		try {

			LPLog.INSTANCE.d(TAG, "getOutputStreamFromBitmap: file extension: " + imageTypeExtension);
			Bitmap.CompressFormat format = Bitmap.CompressFormat.JPEG;
			if (imageTypeExtension.toLowerCase().equals(PNG)) {
				format = Bitmap.CompressFormat.PNG;
			}

			LPLog.INSTANCE.d(TAG, "getOutputStreamFromBitmap: compressing bitmap to byte array");
			ByteArrayOutputStream os = new ByteArrayOutputStream();
			// Use the compress method on the BitMap object to write image to the OutputStream
			bitmapImage.compress(format, compression, os);
			byteArray = os.toByteArray();
			os.close();

		} catch (IOException e) {
			LPLog.INSTANCE.w(TAG, "getOutputStreamFromBitmap: ", e);
		}

		return byteArray;
	}

	/**
	 * Convert the given base64 to image byte array and save it to disk with random generated name
	 */
	public static String saveBase64ToDisk(Context context, String base64, String brandId) {

		if (base64 != null) {

			// Convert the base64 yo byte array
			byte[] byteArray = base64ToByteArray(base64);
			if (byteArray != null) {
				return saveBitmapToDisk(context, byteArray, brandId, ImageFolderType.PREVIEW, null);
			}
		}

		return null;
	}

	/**
	 * Generates a random filename and save it to disk
	 *
	 * @return path to saved file
	 */
	public static String saveBitmapToDisk(Context context, byte[] imageByteArray, String brandId, ImageFolderType imageFolderType, String fileExtension) {

		if (fileExtension == null) {
			fileExtension = ImageUtils.JPG;
		}

		// Generate a random name
		String filename = UUID.randomUUID().toString() + "." + fileExtension;

		return saveBitmapToDisk(context, imageByteArray, brandId, filename, imageFolderType);
	}

	/**
	 * Save the given byte array to disk to location according to imageFolderType
	 *
	 * @return path to saved file
	 */
	private static String saveBitmapToDisk(Context context, byte[] imageByteArray, String brandId, String filename, ImageFolderType imageFolderType) {

		String imageTypeFolder;
		switch (imageFolderType) {
			case PREVIEW:
				imageTypeFolder = PREVIEW_IMAGE_FOLDER;
				break;
			case FULL:
				imageTypeFolder = FULL_IMAGE_FOLDER;
				break;
			default:
				imageTypeFolder = "";
		}

		String path = "/" + IMAGES_FOLDER + brandId + "/" + imageTypeFolder;

		File filePath = FileUtils.getFilePath(context, path, filename);

		if (filePath == null) {
			return null;
		}

		return saveBitmapToDisk(imageByteArray, filePath);
	}

	/**
	 * Save the given byte array to disk to the given file
	 *
	 * @return path to saved file
	 */
	public static String saveBitmapToDisk(byte[] imageByteArray, File filePath) {


		LPLog.INSTANCE.d(TAG, "saveBitmapToDisk: filePath: " + filePath.getAbsolutePath());

		try (FileOutputStream fos = new FileOutputStream(filePath)) {
			fos.write(imageByteArray);
		} catch (FileNotFoundException e) {
			LPLog.INSTANCE.e(TAG, ERR_00000019, "saveBitmapToDisk: File not found", e);
			return null;
		} catch (IOException e) {
			LPLog.INSTANCE.e(TAG, ERR_0000001A, "saveBitmapToDisk: IOException", e);
		}

		LPLog.INSTANCE.d(TAG, "saveBitmapToDisk: file absolute path: " + filePath.getAbsolutePath());
		return filePath.getAbsolutePath();
	}

	/**
	 * Create image file (used to provide to camera) and return its Uri
	 */
	public static Uri createImageFileForCamera(Context context, String brandId) throws IOException {

		File lp_images_folder = new File(context.getFilesDir(), ImageUtils.IMAGES_FOLDER + brandId + "/" + ImageUtils.FULL_IMAGE_FOLDER);

		//noinspection ResultOfMethodCallIgnored
		lp_images_folder.mkdirs();

		return FileProvider.getUriForFile(context,
			   Infra.getFileProviderAuthorityPrefix() + context.getApplicationInfo().packageName, // The prefix here is the same as defined in the provider in the manifest
			   File.createTempFile("image", ".jpg", lp_images_folder));
	}

	private static int decodeExifOrientation(int orientation, boolean fromCamera) {
		int degrees;
		switch (orientation) {
			case ExifInterface.ORIENTATION_ROTATE_90:
				degrees = 90;
				break;
			case ExifInterface.ORIENTATION_ROTATE_180:
				degrees = 180;
				break;
			case ExifInterface.ORIENTATION_ROTATE_270:
				degrees = 270;
				break;
			default:
				degrees = 0;
				break;
		}

		degrees = handleSpecificDevices(fromCamera, degrees);
		return degrees;
	}

	/**
	 * If required, change the rotation degrees according to device and according to whether the image if from camera or not (gallery)
	 * @param fromCamera true - image from camera, false - from gallery
	 * @param degrees degrees to manipulate
	 * @return the manipulated degrees
	 */
	private static int handleSpecificDevices(boolean fromCamera, int degrees) {

		String manufacturer = Build.MANUFACTURER;

		// If the image if from a Samsung gallery, the degrees should be 0 (no rotation)
		if (!fromCamera && manufacturer.equalsIgnoreCase(SAMSUNG)) {
			degrees = 0;
		}

		return degrees;
	}

	/**
	 * creates image on canvas the size of the specific screen size.
	 * @param context the context of caller.
	 * @param windowManager used to get size of screen.
	 * @param myImage the image from resources used as the background image.
	 * @return Drawable which will be set as backgroundDrawable.
	 */
	public static Drawable createImageOnCanvas(Context context, WindowManager windowManager, Bitmap myImage) {
		//getting the height of the actionBar (toolbar) to calculate the size of the picture.
		int actionBarHeight;
		final TypedArray styledAttributes = context.getTheme().obtainStyledAttributes(
				new int[] { android.R.attr.actionBarSize }
		);
		actionBarHeight = (int) styledAttributes.getDimension(0, 0);
		styledAttributes.recycle();

		// getting screen size. height and width.
		DisplayMetrics displayMetrics = new DisplayMetrics();
		windowManager.getDefaultDisplay().getMetrics(displayMetrics);
		int height = displayMetrics.heightPixels;
		int width = displayMetrics.widthPixels;

		//resizing the bitmap image to the appropriate size.
		Bitmap resizedBitmap = Bitmap.createScaledBitmap(
				myImage, width, height - actionBarHeight, false);

		//creating bitmap that will hold the canvas.
		Bitmap result = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
		Canvas canvas = new Canvas(result);

		//drawing the image at the starting point under the actionBar height.
		canvas.drawBitmap(resizedBitmap, width-resizedBitmap.getWidth(), height - resizedBitmap.getHeight(), null);

		//converting bitmap into drawable and return.
		return new BitmapDrawable(getResources(), result);
	}

}
