package com.liveperson.infra;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.LocalBroadcast;


/**
 * Created by Ofir Antebi on 12/9/2015.
 * Service that keep tracking the internet connection
 * and informs the observers when there is a change
 * <p/>
 * Note: In order to be informed about network state change
 * there is a need to register for local broadcasts
 * BROADCAST_INTERNET_CONNECTION_CONNECTED OR BROADCAST_INTERNET_CONNECTION_DISCONNECTED
 */
public class InternetConnectionService implements ShutDown {

    private static final String TAG = InternetConnectionService.class.getSimpleName();
    public static final String BROADCAST_INTERNET_CONNECTION_CONNECTED = "CONNECTION_CONNECTED";
    public static final String BROADCAST_INTERNET_CONNECTION_DISCONNECTED = "CONNECTION_DISCONNECTED";
    private ConnectionReceiver mConnectionReceiver;

    /**
     * Package private
     * Only the infra controller can create it
     */
    InternetConnectionService() {
        registeredReceiver();
    }

    public void registeredReceiver() {
        if (mConnectionReceiver == null){
            mConnectionReceiver = new ConnectionReceiver();
            LPMobileLog.d(TAG, "creating new receiver");
        }
        mConnectionReceiver.register(Infra.instance.getApplicationContext());
    }
    /**
     * Unregister the receiver from being informed about network changes
     */
    public void unRegisteredReceiver() {
        if (mConnectionReceiver != null){
            LPMobileLog.d(TAG, "un-registering the receiver");
            try {
                Infra.instance.getApplicationContext().unregisterReceiver(mConnectionReceiver);
            } catch (IllegalArgumentException e) {
                LPMobileLog.e(TAG, "Failed to un-register connection receiver. Reason: " + e);
            }
            mConnectionReceiver = null;
        }
    }

    /**
     * Check if there is available internet connection
     *
     * @return if network is available
     */
    public static boolean isNetworkAvailable() {

        Context applicationContext = Infra.instance.getApplicationContext();
        if (applicationContext != null){
            ConnectivityManager connectivityManager = (ConnectivityManager) applicationContext.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            boolean b = networkInfo != null && networkInfo.isConnectedOrConnecting();
            LPMobileLog.d(TAG, "isNetworkAvailable: isConnected = " + b );
            LPMobileLog.d(TAG, "isNetworkAvailable: networkInfo = " + networkInfo );
            return b;

        }
        LPMobileLog.d(TAG, "isNetworkAvailable: app context is null!" );
        return false;
    }

    @Override
    public void shutDown() {
        unRegisteredReceiver();
    }


    /**
     * The receiver that gets the broadcasts from the system
     */
    private class ConnectionReceiver extends BroadcastReceiver {

        private final IntentFilter intentFilter;

        private boolean lastConnectionState;

        public ConnectionReceiver(){
            intentFilter = new IntentFilter();
            intentFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
        }

        public void register(Context context) {
            Intent currentStatus = context.registerReceiver(this, intentFilter);
            lastConnectionState = isNetworkAvailable();
            onReceive(context, currentStatus);
            //sendUpdateIntent(lastConnectionState);
            LPMobileLog.d(TAG, "RegisteredReceiver, currentStatus = " + currentStatus);
            LPMobileLog.d(TAG, "RegisteredReceiver, lastConnectionState = " + lastConnectionState);
        }

        @Override
        public void onReceive(Context context, Intent intent) {
            if (intent != null) {
                String action = intent.getAction();
                if (action.equals(ConnectivityManager.CONNECTIVITY_ACTION)) {
                    boolean isConnected = isNetworkAvailable();
                    LPMobileLog.d(TAG, "isConnected " + isConnected);
                    LPMobileLog.d(TAG, "lastConnectionState " + lastConnectionState);
                    if (lastConnectionState != isConnected) {
                        lastConnectionState = isConnected;
                        sendUpdateIntent(isConnected);
                        LPMobileLog.i(TAG, "Getting broadcast with action " + intent.getAction() +", connected = "+ lastConnectionState);
                    }
                }
            }
        }

        private void sendUpdateIntent(boolean isConnected) {
            String intentAction;
            if (isConnected) {
                intentAction = BROADCAST_INTERNET_CONNECTION_CONNECTED;
            } else {
                intentAction = BROADCAST_INTERNET_CONNECTION_DISCONNECTED;
            }
            LocalBroadcast.sendBroadcast(intentAction);
        }

    }
}
