package com.liveperson.infra.network.socket;

import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.log.LPMobileLog;

import java.util.Arrays;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by Ilya Gazman on 11/2/2015.
 *
 * Mapping responses to requests
 */
public class ResponseMap implements ShutDown {

    private static final String TAG = ResponseMap.class.getSimpleName();
    public static final String RESPONSE_TAG = "FLOW_RESPONSES_";
    //    private HashMap<String, Class<? extends BaseResponseHandler>> mGeneralMap = new HashMap<>();
//    private HashMap<Long, BaseResponseHandler> uIDMap = new HashMap<>();
    //<response type, matching response> //OnlineEventDistribution, Notification..
    private ConcurrentHashMap <String, Class<? extends BaseResponseHandler>> mGeneralMap = new ConcurrentHashMap<>();
    //<request id, matching response>
    private ConcurrentHashMap<Long, BaseResponseHandler> uIDMap = new ConcurrentHashMap<>();
    private GeneralResponseHandler generalResponseHandler = null;

    public ResponseMap() {}

    /**
     * Maps Response to request id
     * @param requestId a unique id associated with request
     * @param handler the response handler
     */
    public void putRequestIdHandler(final long requestId, final BaseResponseHandler handler) {
        if (uIDMap.containsKey(requestId)){
            uIDMap.remove(requestId);
        }
        uIDMap.put(requestId, handler);
    }

    /**
     * Maps server response message type to response handler
     * @param messageType server response message type
     * @param handler response handler
     */
    public void putGeneralHandler(String messageType, Class<? extends BaseResponseHandler> handler) {
        if (mGeneralMap.containsKey(messageType)){
            mGeneralMap.remove(messageType);
        }
        mGeneralMap.put(messageType, handler);
    }


    public void onRequestHandled(long requestId) {
        if (uIDMap.containsKey(requestId)){
            uIDMap.remove(requestId);
            LPMobileLog.d(TAG, "onRequestHandled: " + requestId + ", removing it from map. map = " + Arrays.toString(uIDMap.keySet().toArray()));
        }else{
            LPMobileLog.w(TAG, "onRequestHandled NOT IN MAP! " + requestId + ", map = " + Arrays.toString(uIDMap.keySet().toArray()));
        }
    }

    public void onSocketClosed() {
        LPMobileLog.d(TAG, "onSocketClosed: sending event to all waiting requests. map = " + Arrays.toString(uIDMap.keySet().toArray()));
        for (BaseResponseHandler response: uIDMap.values()) {
            response.giveUp();
        }
        uIDMap.clear();
    }

    /**
     * Return response handler
     * @param requestId
     * @return
     */
    public BaseResponseHandler getRequestIdHandler(String messageType, long requestId) {
        BaseResponseHandler baseResponseHandler = uIDMap.get(requestId);
        if (baseResponseHandler == null){
            LPMobileLog.d(RESPONSE_TAG+ TAG,"Getting general response for message type :" + messageType);
            return getGeneralHandler(messageType);
        }else{
            LPMobileLog.d(RESPONSE_TAG+ TAG,"Found response in map :" + messageType  +  " requestId = " + requestId);
            return baseResponseHandler;
        }
    }

    /**
     * Creates a proper general handler by a given messageType
     * @param messageType the message type
     * @return BaseResponseHandler
     */
    private BaseResponseHandler getGeneralHandler(String messageType) {
        if (generalResponseHandler == null){
            return null;
        }
        return generalResponseHandler.createInstance(messageType);
    }

    public void putGeneralHandler(GeneralResponseHandler generalResponseHandler) {
        this.generalResponseHandler = generalResponseHandler;
    }
    /**
     * clear the mapped requests
     */
    @Override
    public void shutDown() {
        onSocketClosed();
        uIDMap.clear();
        mGeneralMap.clear();
    }

}
