package com.liveperson.infra.utils;

import android.util.Log;

import com.liveperson.infra.log.LPMobileLog;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * Created by ofira on 06/12/2015.
 * Thread pool with max size of 5
 * Create new thread when there is a need until a total of 5 threads
 * Each thread is waiting 60 seconds for a job and terminates in case there is no job
 * to execute
 */
public class ThreadPoolExecutor {

    private static final String TAG = ThreadPoolExecutor.class.getSimpleName();
    private static final int THREAD_POOL_SIZE = 5;
    private static final long THREAD_KEEP_ALIVE_TIME = 60L;
    private static ScheduledThreadPoolExecutor mScheduledExecutor;
    private static ExecutorService mExecutor;

    private static void init() {
        if (mExecutor == null) {
            mExecutor = Executors.newFixedThreadPool(THREAD_POOL_SIZE);
        }
    }

    private static void initScheduled() {
        if (mScheduledExecutor == null) {
            //https://docs.oracle.com/javase/7/docs/api/java/util/concurrent/ScheduledThreadPoolExecutor.html
            mScheduledExecutor = new ScheduledThreadPoolExecutor(1);
            mScheduledExecutor.setKeepAliveTime(THREAD_KEEP_ALIVE_TIME, TimeUnit.SECONDS);
        }
    }


    /**
     * Execute the runnable immediately
     *
     * @param runnable
     */
    public static void execute(Runnable runnable) {
        init();
        mExecutor.execute(runnable);
    }


    /**
     * Execute the runnable with a delay
     *
     * @param runnable
     * @param delay    - delay in milliseconds
     */
    public static void executeDelayed(Runnable runnable, long delay) {
        if (delay == 0){
            execute(runnable);
        }else{
            initScheduled();
            mScheduledExecutor.schedule(runnable, delay, TimeUnit.MILLISECONDS);
        }
    }

    public static void killAll() {

        try {
            if (mScheduledExecutor != null) {

                LPMobileLog.d(TAG, "Start killing mScheduledExecutor");
                //Cancel scheduled but not started task, and avoid new ones
                mScheduledExecutor.shutdown();

                //Wait for the running tasks
                mScheduledExecutor.awaitTermination(1, TimeUnit.SECONDS);

                //Interrupt the threads and shutdown the scheduler
                mScheduledExecutor.shutdownNow();
                LPMobileLog.d(TAG, "Finished killing mScheduledExecutor");

            }

            if (mExecutor != null) {

                LPMobileLog.d(TAG, "Start killing mExecutor");
                //Cancel scheduled but not started task, and avoid new ones
                mExecutor.shutdown();

                //Wait for the running tasks
                mExecutor.awaitTermination(1, TimeUnit.SECONDS);

                //Interrupt the threads and shutdown the scheduler
                mExecutor.shutdownNow();
                LPMobileLog.d(TAG, "Finished killing mExecutor");

            }
        } catch (InterruptedException e) {
            LPMobileLog.w(TAG, "Error killing Executors");
        } finally {
            mScheduledExecutor = null;
            mExecutor = null;
        }
    }
}