package com.liveperson.infra;

import android.os.Bundle;
import android.text.TextUtils;

import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.LocalBroadcast;

import java.util.HashSet;

/**
 * Created by shiranr on 12/11/2015.
 * Modified by nirni
 * <p/>
 * This is used to register any view as the one in the foreground. The view that registered needs
 * to provide the brand ID which it uses.
 * <p/>
 * When a view is registered/unregistered a corresponding broadcast is sent.
 */
public class ForegroundService implements ShutDown, ForegroundServiceInterface {

    private static final String TAG = ForegroundService.class.getSimpleName();
    public static final String SCREEN_FOREGROUND_ACTION = "SCREEN_FOREGROUND_ACTION";
    public static final String SCREEN_FOREGROUND_EXTRA_BRAND_ID = "brand_id";
    public static final String SCREEN_FOREGROUND_EXTRA_TARGET_ID = "target_id";
    public static final String SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND = "is_foreground";
    private HashSet<String> mForegroundBrands = new HashSet<>();
    private HashSet<String> mForegroundTargets = new HashSet<>();

    private static volatile ForegroundService Instance = null;


    private ForegroundService(){}


    /**
     * Get instance of the infra Preference Manager
     * Note: this class uses the application context initialized in the {@link Infra} class
     *
     * @return
     */
    public static ForegroundService getInstance() {
        if (Instance == null) {
            synchronized (ForegroundService.class) {
                if (Instance == null) {
                    Instance = new ForegroundService();
                }
            }
        }
        return Instance;
    }


    /**
     * Register a view for the given brand as foreground
     *
     * @param brandId
     */
    @Override
    public void register(String brandId, String targetId) {
        LPMobileLog.i(TAG, "register with brand " + brandId);
        boolean needBroadcast = false;
        if (!TextUtils.isEmpty(targetId) && !mForegroundTargets.contains(targetId)) {
            mForegroundTargets.add(targetId);
            needBroadcast = true;
        }
        if (!TextUtils.isEmpty(brandId) && !mForegroundBrands.contains(brandId)) {
            mForegroundBrands.add(brandId);
            needBroadcast = true;
        }
        if(needBroadcast) {
            Bundle bundle = new Bundle();
            bundle.putString(SCREEN_FOREGROUND_EXTRA_BRAND_ID, brandId);
            bundle.putString(SCREEN_FOREGROUND_EXTRA_TARGET_ID, targetId);
            bundle.putBoolean(SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND, true);
            LocalBroadcast.sendBroadcast(SCREEN_FOREGROUND_ACTION, bundle);
        }
    }

    /**
     * Unregister a view for the given brand as background
     *
     * @param brandId
     */
    @Override
    public void unregisterBrand(String brandId) {
        LPMobileLog.i(TAG, "unregister with brand " + brandId);
        if (!TextUtils.isEmpty(brandId) && mForegroundBrands.contains(brandId)) {
            mForegroundBrands.remove(brandId);

            Bundle bundle = new Bundle();
            bundle.putString(SCREEN_FOREGROUND_EXTRA_BRAND_ID, brandId);
            bundle.putBoolean(SCREEN_FOREGROUND_EXTRA_IS_FOREGROUND, false);
            LocalBroadcast.sendBroadcast(SCREEN_FOREGROUND_ACTION, bundle);
        }
    }

    @Override
    public void unregisterAll() {
        for(String brand : mForegroundBrands){
            unregisterBrand(brand);
        }
        for(String target : mForegroundTargets){
            unregisterBrand(target);
        }
        mForegroundBrands.clear();

    }


    public void restart() {
        HashSet<String> tempForegroundBrands = (HashSet<String>) mForegroundBrands.clone();
        mForegroundBrands.clear();
        for(String brand : tempForegroundBrands){
            register(brand, null);
        }
    }
    /**
     * Unregister a view for the given brand as background
     *
     * @param targetId
     */
    @Override
    public void unregisterTarget(String targetId) {
        LPMobileLog.i(TAG, "unregister with target " + targetId);
        if (!TextUtils.isEmpty(targetId) && mForegroundTargets.contains(targetId)) {
            mForegroundTargets.remove(targetId);
        }
    }


    @Override
    public boolean isBrandForeground() {
        return (!mForegroundBrands.isEmpty());
    }

    @Override
    public boolean isBrandForeground(String brandId) {
        return mForegroundBrands.contains(brandId);
    }

    @Override
    public boolean isTargetForeground(String brandId) {
        return mForegroundTargets.contains(brandId);
    }

    @Override
    public HashSet<String> getForegroundBrandId() {
        return mForegroundBrands;
    }

    @Override
    public void shutDown() {
        unregisterAll();
    }
}
