package com.liveperson.infra.log

/**
 * Standard interface for all system logging implementations. Most classes should use
 * [AndroidLogger] as the default concrete implementation, and allow overrides
 * with a custom logging implementation for unit testing purposes.
 */
interface Logger {

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun v(tag: String, message: String)

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun v(tag: String, message: String, exception: Throwable)

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun d(tag: String, message: String)

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun d(tag: String, message: String, exception: Throwable)

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun i(tag: String, message: String)

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun i(tag: String, message: String, exception: Throwable)

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun w(tag: String, message: String)

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun w(tag: String, message: String, exception: Throwable)

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun e(tag: String, message: String)

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun e(tag: String, message: String, exception: Throwable)

	/**
	 * Logs a message using the highest possible log-level (WHAT_A_TERRIBLE_FAILURE).
	 * This information should be understandable by businesspeople.
	 * Use **only** for complete failures in logic that should never happen ever.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun wtf(tag: String, message: String)

	/**
	 * Logs a message using the highest possible log-level (WHAT_A_TERRIBLE_FAILURE).
	 * This information should be understandable by businesspeople.
	 * Use **only** for complete failures in logic that should never happen ever.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun wtf(tag: String, message: String, exception: Throwable)
}
