package com.liveperson.infra.sdkstatemachine;

import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.sdkstatemachine.events.InitEvent;
import com.liveperson.infra.sdkstatemachine.events.LogoutEvent;
import com.liveperson.infra.sdkstatemachine.events.PreLogoutCompletedEvent;
import com.liveperson.infra.sdkstatemachine.events.ShutDownCompletedEvent;
import com.liveperson.infra.sdkstatemachine.events.ShutDownEvent;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.BaseStateMachine;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;

/**
 * Created by shiranr on 04/05/2016.
 */
public class InfraStateMachine extends BaseStateMachine {
    private static final String TAG = InfraStateMachine.class.getSimpleName();
    //STATES//
    private final Idle mIdleState = new Idle("IdleState", TAG + "_Idle");
    private final Initialized mInitializedState = new Initialized("Initialized", TAG);
    private final Initializing mInitializingState = new Initializing("Initializing", TAG);
    private final ShuttingDown mShuttingDown = new ShuttingDown("ShuttingDown", TAG);
    private final Logout mLoggingOutState = new Logout("Logout", TAG);

    public InfraStateMachine() {
        super(TAG);
        initActiveState(mIdleState);
    }

    public void initSDK(InitProcess initProcess) {
        if (initProcess == null) {
            LPMobileLog.e(TAG, "getInitProcess() is null!! error!");
            return;
        }
        postEvent(new InitEvent(initProcess));
    }

    public void shutDownSDK(ShutDownProcess shutDownProcess) {
        if (shutDownProcess == null) {
            LPMobileLog.e(TAG, "getInitProcess() is null!! error!");
            return;
        }
        postEvent(new ShutDownEvent(shutDownProcess));
    }

    public void logoutSDK(LogoutProcess logoutProcess) {
        if (logoutProcess == null) {
            LPMobileLog.e(TAG, "logoutProcess is null!! error!");
            return;
        }
        postEvent(new LogoutEvent(logoutProcess));
    }

    public boolean isSDKInitialized() {
        return (activeState()!= null && activeState().equals(mInitializedState));
    }

    /**
     * Idle- not initialized yet.
     */
    class Idle extends BaseInfraState{
        public Idle (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            changeStateAndPassEvent(mInitializingState, initEvent);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            changeStateAndPassEvent(mLoggingOutState, logoutEvent);
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            //ignore
            LPMobileLog.d(TAG, "shutDownEvent on init. ignore.");
        }
    }

    class Initializing extends InProgressState{

        public Initializing (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            InitProcess initProcess = initEvent.getInitProcess();

            try {
                initProcess.init();
                LPMobileLog.d(TAG, "Init finished Successfully! :) ");
                changeStateAndPassEvent(mInitializedState, initEvent);
            } catch (Exception e) {
                LPMobileLog.e(TAG, "Exception while visiting InitEvent: " + initEvent, e);
                initProcess.getInitCallback().onInitFailed(e);
                changeState(mIdleState);
            }
        }
    }
    class Initialized extends BaseInfraState{
        public Initialized (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            LPMobileLog.d(TAG, "Initialized! calling onInitSucceed callback");
            initEvent.getInitProcess().getInitCallback().onInitSucceed();
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            changeStateAndPassEvent(mShuttingDown, shutDownEvent);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            logoutEvent.setSkipInit(true);
            changeStateAndPassEvent(mLoggingOutState, logoutEvent);
        }
    }

    /**
     * Shutting down the SDK
     */
    class ShuttingDown extends InProgressState{

        private ShutDownProcess mShutDownProcess = null;
        private InitEvent mInitEventAfterShutDown = null;

        public ShuttingDown (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mShutDownProcess = null;
            mInitEventAfterShutDown = null;
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            //call super for logout event handling
            super.visit(shutDownEvent);
            //first time in this state
            if (mShutDownProcess == null){
                mShutDownProcess = shutDownEvent.getInitProcess();
                mShutDownProcess.shutDown(new ShutDownCompletionListener() {
                    @Override
                    public void shutDownCompleted() {
                        postEvent(new ShutDownCompletedEvent());
                    }

					@Override
					public void shutDownFailed() {

					}
				});
            }else{
                //already during shut down
                //shut down reset waiting init event
                LPMobileLog.d(TAG, "got shutDown event while processing Shutting Down... removing waiting init event if exists.");
                mInitEventAfterShutDown = null;
            }
        }

        @Override
        public void visit(ShutDownCompletedEvent shutDownCompletedEvent) {
            LPMobileLog.d(TAG, "Shut down finished successfully! :) ");
            mShutDownProcess.getShutDownCallback().onShutDownSucceed();
            if (logoutEventWaiting != null){
                LPMobileLog.d(TAG, "Logout event waiting, logging out...");
                changeStateAndPassEvent(mLoggingOutState, logoutEventWaiting);
            }else if (mInitEventAfterShutDown != null){
                LPMobileLog.d(TAG, "Init event waiting, moving to initialized...");
                changeStateAndPassEvent(mInitializingState, mInitEventAfterShutDown);
            }else{
                changeState(mIdleState);
            }
        }

        @Override
        public void visit(InitEvent initEvent) {
            super.visit(initEvent);
            LPMobileLog.d(TAG, "got init event while processing Shutting Down...");
            mInitEventAfterShutDown = initEvent;
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            super.visit(logoutEvent);
            LPMobileLog.d(TAG, "Got logout event while processing Shutting Down... removing init waiting event");
            mInitEventAfterShutDown = null;
        }
    }

    /**
     * Logging out from SDK - removing all user data
     */
    class Logout extends BaseInfraState {
        private InitEvent mInitEventAfterLogout = null;
        private LogoutEvent mLogoutEvent;

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mInitEventAfterLogout = null;
            mLogoutEvent = null;
        }

        public Logout (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            if (mLogoutEvent == null){
                mLogoutEvent = logoutEvent;
                mInitEventAfterLogout = logoutEvent.getInitEvent();
                logout();
            }else{
                //ignore
                LPMobileLog.d(TAG, "got logout event while processing logout.. ignoring event");
            }
        }

        @Override
        public void visit(InitEvent initEvent) {
            mInitEventAfterLogout = initEvent;
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            mInitEventAfterLogout = null;
        }

        @Override
        public void visit(PreLogoutCompletedEvent preLogoutCompletedEvent) {
            //shut down
            LPMobileLog.d(TAG, "shutDownForLogout...");
            LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    postEvent(new ShutDownCompletedEvent());
                }

				@Override
				public void shutDownFailed() {

				}
			});
        }

        @Override
        public void visit(ShutDownCompletedEvent shutDownCompletedEvent) {
            //logout
            LPMobileLog.d(TAG, "logout...");
            LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            logoutProcess.logout();
            onLogoutSucceed(logoutProcess);
        }

        private void logout() {
            LPMobileLog.d(TAG, "Stating logout process...");
            final LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            try {
                if(!mLogoutEvent.skipInit()){
                    //init if needed
                    LPMobileLog.d(TAG, "initForLogout...");
                    logoutProcess.initForLogout();
                }

                // pre logout
                LPMobileLog.d(TAG, "preLogout...");
                logoutProcess.preLogout(new PreLogoutCompletionListener() {
                    @Override
                    public void preLogoutCompleted() {
                        postEvent(new PreLogoutCompletedEvent());
                    }

                    @Override
                    public void preLogoutFailed(Exception e) {
                        //notify listener logout failed.
                        LPMobileLog.w(TAG, "error while preLogoutFailed: ", e);
                        onLogoutFailed(e, logoutProcess);

                    }
                });

            } catch (Exception e) {
                LPMobileLog.w(TAG, "error while logout: ", e);
                onLogoutFailed(e, logoutProcess);
            }
        }

        private void onLogoutSucceed(LogoutProcess logoutProcess) {
            logoutProcess.getLogoutCallback().onLogoutSucceed();
            exitState();
        }

        private void onLogoutFailed(Exception e, LogoutProcess logoutProcess) {
            logoutProcess.getLogoutCallback().onLogoutFailed(e);
            exitState();
        }

        private void exitState() {
            if (mInitEventAfterLogout != null) {
                changeStateAndPassEvent(mIdleState, mInitEventAfterLogout);
            }else{
                changeState(mIdleState);
            }
        }

    }


    abstract class InProgressState extends BaseInfraState{
        LogoutEvent logoutEventWaiting = null;

        public InProgressState (String name, String logTag) {
            super( name, logTag);
        }

        @Override
        public void actionOnEntry() {
            logoutEventWaiting = null;
        }

        @Override
        public void visit(InitEvent initEvent) {
            if (logoutEventWaiting != null){
                logoutEventWaiting.setInitAfterLogout(initEvent);
                LPMobileLog.d(TAG, "Logout event waiting, init after logout is NOT null ");
            }
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            if (logoutEventWaiting != null){
                logoutEventWaiting.setInitAfterLogout(null);
                LPMobileLog.d(TAG, "Logout event waiting, init after logout is null ");
            }
        }
        @Override
        public void visit(LogoutEvent logoutEvent) {
            logoutEventWaiting = logoutEvent;
            logoutEventWaiting.setInitAfterLogout(null);
            LPMobileLog.d(TAG, "Logout event waiting, init after logout is null ");
        }
    }
}
