package com.liveperson.infra;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.VisibleForTesting;
import android.text.TextUtils;

import com.liveperson.infra.callbacks.InitLivePersonCallBack;
import com.liveperson.infra.callbacks.LogoutLivePersonCallBack;
import com.liveperson.infra.configuration.LptagEnvironment;
import com.liveperson.infra.controller.DBEncryptionKeyHelper;
import com.liveperson.infra.database.DatabaseManager;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.log.logreporter.loggos.Loggos;
import com.liveperson.infra.managers.PreferenceManager;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.socket.SocketManager;
import com.liveperson.infra.sdkstatemachine.InfraStateMachine;
import com.liveperson.infra.sdkstatemachine.init.InfraInitData;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;
import com.liveperson.infra.statemachine.StateMachineExecutor;
import com.liveperson.infra.utils.LocaleUtils;
import com.liveperson.infra.utils.ThreadPoolExecutor;

/**
 * Created by Ilya Gazman on 11/3/2015.
 */
public enum Infra implements Clearable {
    instance;

    private static final String TAG = Infra.class.getSimpleName();
    public static final String KEY_BRAND_ID = "brand_id";
    public static final String KEY_AUTH_KEY = "auth_key";
    public static final String KEY_READ_ONLY = "read_only";
    public static final String KEY_VIEW_PARAMS = "view_params";

    public static final String KEY_TARGET_ID = "target_id";
    public static final String SDK_VERSION = "SDK_VERSION";
    private static final String FILE_PROVIDER_AUTHORITY_PREFIX = "com.liveperson.infra.provider."; // private since we don't want direct access to it

    private InternetConnectionService mConnectionService;
    private Context mAppContext;
    private Handler mAppHandler;
    private DBEncryptionKeyHelper dbEncryptionKeyHelper;
    private String mHostVersion;
    private Loggos mLoggos;

	/**
	 * LPTag environment that holds the relevant LPTag domain.
	 *
	 * Since LPTag uses different domain for each environment (GA, Alpha, QA), we need a way to set
	 * the environment that we work with from test host app.
	 * The default setting in this is GA (Production), but if a test app need to set it for different
	 * environment it can use the setEnvironment() method to set a different environment.
 	 */
	private LptagEnvironment mLptagEnvironment;

    InfraStateMachine stateMachine = null;

    Infra() {
        mAppHandler = new Handler(Looper.getMainLooper());
        initStateMachine();
    }

    /**
     * Will run the init process when system is starting
     */
    public void init(final Context context, final InfraInitData initData, final InitProcess entryPoint) {
        final InitProcess infraInit = new InfraInitProcess(context, initData, entryPoint);

        boolean interceptorsEnabled = context.getResources().getBoolean(R.bool.lp_interceptors_enabled);
		LPMobileLog.d(TAG, "init: Interceptors enabled: " + interceptorsEnabled);
		// Set the interceptors only if it is enabled and interceptors provided
		if (interceptorsEnabled && (initData.getInterceptors() != null)) {
			HttpHandler.addInterceptors(initData.getInterceptors().getHttpInterceptorList());
		}

		stateMachine.initSDK(infraInit);
    }

    private void initStateMachine() {
        if (stateMachine == null) {
            stateMachine = new InfraStateMachine();
            stateMachine.setStateMachineExecutor(new StateMachineExecutor(stateMachine.getTag(), stateMachine));
        }
    }


    /**
     * init the member mHostVersion and save it in Shared Preferences.
     * @param sdkVersion - if null, mHostVersion will be initiate from the saved value in Shared Pref.
     * Throw IllegalStateException in case sdkVersion is null and there is no value in Shared Pref.
     */
    private void initSDKVersion(String sdkVersion) {
        if (TextUtils.isEmpty(sdkVersion)){
            //read from sharedPref
            mHostVersion = PreferenceManager.getInstance().getStringValue(SDK_VERSION, "" , "");
            if (TextUtils.isEmpty(mHostVersion)){
                throw new IllegalStateException("must have a value for sdk-version!");
            }
        }else{
            mHostVersion = sdkVersion;
            //save in sharedPref
            PreferenceManager.getInstance().setStringValue(SDK_VERSION, "", mHostVersion);
        }
    }


    public boolean isInitialized() {
        return stateMachine != null && stateMachine.isSDKInitialized();
    }

    private void initInfra(Context context, InfraInitData initData) {
        setContext(context);
        initSDKVersion(initData != null ? initData.getSdkVersion() : null);


        mLoggos = new Loggos();
		mLptagEnvironment = new LptagEnvironment();
        if (dbEncryptionKeyHelper == null){
            dbEncryptionKeyHelper = new DBEncryptionKeyHelper(null);
        }
        mConnectionService = new InternetConnectionService();
    }


    /**
     * Shutting down the sdk. will shut down infra and the given ShutDownProcess.
     * @param shutDownProcess - the upper process to be shut down.
     */
    public void shutDown(final ShutDownProcess shutDownProcess) {
        final ShutDownProcess shutDownInfra = new InfraShutDownProcess(shutDownProcess);
        stateMachine.shutDownSDK(shutDownInfra);
    }

    public void restart(){
       /* LPMobileLog.d(TAG, "Restarting...");
*//*
        ForegroundService.getInstance().shutDown();
*//*
        ThreadPoolExecutor.killAll();
        DatabaseManager.getInstance().shutDown();
        SocketManager.getInstance().shutDown();
        //mAppContext = null;
        mAppHandler.removeCallbacksAndMessages(null);
        mAppHandler = null;
        mAppHandler = new Handler(Looper.getMainLooper());
        if (mEntryProcess != null){
//            mEntryProcess.restart();
        }
        ForegroundService.getInstance().restart();*/
    }


    private void infraShutDown() {
        ForegroundService.getInstance().shutDown();
        SocketManager.getInstance().shutDown();
        ThreadPoolExecutor.killAll();
        DatabaseManager.getInstance().shutDown();
        mConnectionService.shutDown();
        //mAppContext = null;
        mAppHandler.removeCallbacksAndMessages(null);
        //mAppHandler = null;
        mHostVersion = null;
        //dbEncryptionKeyHelper = null;
    }

    /**
     * @return the context of the application
     */
    public Context getApplicationContext() {
        return mAppContext;
    }

    public void postOnMainThread(Runnable runnable) {
        if (mAppHandler != null) {
            mAppHandler.post(runnable);
        }
    }

    public Handler getApplicationHandler() {
        return mAppHandler;
    }

    /**
     * Should be used only for testing. Example: GetUnreadMessagesCountCommandTest.class
     * @param handler
     */
    @VisibleForTesting
    public void setApplicationHandler(Handler handler) {
        mAppHandler = handler;
    }


    public String getHostVersion() {
        return mHostVersion;
    }


    public void unregisterToNetworkChanges() {
        if (mConnectionService != null){
            mConnectionService.unRegisteredReceiver();
        }
    }

    public void registerToNetworkChanges() {
        if (mConnectionService != null){
            mConnectionService.registeredReceiver();
        }
    }


    /**
     * deleting all the user data from DB and SharedPreferences.
     */
    @Override
    public void clear() {
        if (mAppContext != null) {
            PreferenceManager.getInstance().clearAll();
            DatabaseManager.getInstance().clear();
            dbEncryptionKeyHelper.clear();
            dbEncryptionKeyHelper = null;
        }
    }

    public DBEncryptionKeyHelper getDbEncryptionKeyHelper() {
        return dbEncryptionKeyHelper;
    }

    public Loggos getLoggos() {
        return mLoggos;
    }

	public LptagEnvironment getLptagEnvironment() {
		return mLptagEnvironment;
	}

	/**
     * Logging out infra, deleting user data
     * @param context - app Context.
     * @param initData - initData for logout.
     * @param logoutProcess - upper level logout process.
     */
    public void logout(final Context context, InfraInitData initData, final LogoutProcess logoutProcess) {
        LogoutProcess infraLogoutProcess = new InfraLogoutProcess(context, initData, logoutProcess);
        stateMachine.logoutSDK(infraLogoutProcess);

    }

	public static String getFileProviderAuthorityPrefix(){
		return FILE_PROVIDER_AUTHORITY_PREFIX;
	}

    public void setContext(Context context) {
        if (context != null) {
            mAppContext = context.getApplicationContext();
        } else {
            LPMobileLog.e(TAG, "setContext: The context cannot be null!");
        }
    }

    /**
     * Init Infra with sdk version and than initiating the upper level.
     */
    private class InfraInitProcess extends InitProcess {

        private final Context context;
        private final InfraInitData initData;
        private final InitProcess entryPoint;

        public InfraInitProcess(Context context,  InfraInitData initData, InitProcess entryPoint) {
            this.context = context;
            this.initData = initData;
            this.entryPoint = entryPoint;
        }

        @Override
        public void init() {
            LPMobileLog.d(TAG, "Initializing!");
            initInfra(context, initData);
            entryPoint.init();
        }

        @Override
        public InitLivePersonCallBack getInitCallback() {
            return entryPoint.getInitCallback();
        }
    }

    /**
     * Shutting down the sdk. will shut down the given upper level ShutDownProcess and than shut down infra.
     */
    private class InfraShutDownProcess extends ShutDownProcess {

        //upper level shut down process.
        private final ShutDownProcess shutDownProcess;

        public InfraShutDownProcess(ShutDownProcess shutDownProcess) {
            this.shutDownProcess = shutDownProcess;
        }

        @Override
        public void shutDown(final ShutDownCompletionListener listener) {
            //first, shut down the upper level process.
            //when done, can run infra shutdown.
            shutDownProcess.shutDown(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    LPMobileLog.d(TAG, "Shutting down...");
                    infraShutDown();
                    listener.shutDownCompleted();
                }

				@Override
				public void shutDownFailed() {
					listener.shutDownFailed();
				}
			});
        }
    }

    /**
     * Logging out infra, deleting user data
     */
    private class InfraLogoutProcess extends LogoutProcess {

        private final Context context;
        private final LogoutProcess logoutProcess;
        private final InfraInitData initData;

        public InfraLogoutProcess(Context context, InfraInitData initData, LogoutProcess logoutProcess) {
            this.context = context;
            this.logoutProcess = logoutProcess;
            this.initData = initData;
        }

        /**
         * init sdk for preLogout() processes.
         */
        @Override
        public void initForLogout() {
            initInfra(context, initData);
            logoutProcess.initForLogout();
        }

        /**
         * Things need to be done before shut down & logout.
         * @param listener
         */
        @Override
        public void preLogout(final PreLogoutCompletionListener listener) {
            logoutProcess.preLogout(new PreLogoutCompletionListener() {
                @Override
                public void preLogoutCompleted() {
                    //no action needed in INFRA
                    listener.preLogoutCompleted();
                }

                @Override
                public void preLogoutFailed(Exception e) {
                    listener.preLogoutFailed(e);
                }
            });
        }

        /**
         * shut down Infra before logout.
         * @param listener
         */
        @Override
        public void shutDownForLogout(final ShutDownCompletionListener listener) {
            logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    LPMobileLog.d(TAG, "Shutting down for logout...");
                    infraShutDown();
                    listener.shutDownCompleted();
                }

				@Override
				public void shutDownFailed() {
					listener.shutDownFailed();
				}
			});
        }

        /**
         * logging out Infra after logging out from the upper level.
         * removing DB and shared preferences.
         */
        @Override
        public void logout() {
            logoutProcess.logout();
            clear();
            LocaleUtils.getInstance().clearEngagementLanguageCode();
        }

        @Override
        public LogoutLivePersonCallBack getLogoutCallback() {
            return logoutProcess.getLogoutCallback();
        }
    }
}
