package com.liveperson.infra.database;

import android.content.ContentValues;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;

import com.liveperson.infra.database.transaction_helper.InsertOrUpdateSQLCommand;
import com.liveperson.infra.database.transaction_helper.SQLiteCommand;
import com.liveperson.infra.database.transaction_helper.UpdateSQLCommand;
import com.liveperson.infra.log.LPMobileLog;

import java.util.List;

/**
 * Created by Ilya Gazman on 11/15/2015.
 * <p/>
 * Its a utility class for all the queries to database
 */
public class DBUtilities implements Cloneable {

	private static final String TAG = DBUtilities.class.getSimpleName();
	public static final long ROW_UPDATED = -1;
	private DatabaseManager.DatabaseHelper mDBHelper;
    private String mTableName;

    // protected constructor
    DBUtilities(DatabaseManager.DatabaseHelper dbHelper) {
        mDBHelper = dbHelper;
    }

    /**
     * Obtains table name
     *
     * @param tableName the table name of the current table
     */
    void init(String tableName) {
        mTableName = tableName;
    }

    /**
     * Insert row to selected table
     *
     * @param values
     * @return
     */

    /**
     * SQLite Insert query
     *
     * @param values
     * @return the id of the inserted raw
     */
    public long insert(ContentValues values) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        return db.insert(mTableName, null, values);

    }


    /**
     * SQLite Insert query
     *
     * @param values
     * @return the id of the inserted raw
     */
    public long insertWithOnConflict(ContentValues values) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        return db.insertWithOnConflict(mTableName, null, values, SQLiteDatabase.CONFLICT_IGNORE);

    }

    /**
     * Insert row to selected table
     *
     * @param values
     * @return
     */
    public long replace(ContentValues values) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        return db.replace(mTableName, null, values);

    }

    /**
     * Execute query on selected table
     *
     * @param columns
     * @param selection
     * @param selectionArgs
     * @param groupBy
     * @param having
     * @param orderBy
     * @return
     */
    public Cursor query(String[] columns, String selection, String[] selectionArgs, String groupBy, String having, String orderBy) {
        return query(columns, selection, selectionArgs, groupBy, having, orderBy, null);
    }


    /**
     * Execute query on selected table
     *
     * @param columns
     * @param selection
     * @param selectionArgs
     * @param groupBy
     * @param having
     * @param orderBy
     * @param limit
     * @return
     */
    public Cursor query(String[] columns, String selection, String[] selectionArgs, String groupBy, String having, String orderBy, String limit) {
        SQLiteDatabase db = mDBHelper.getReadableDatabase();
        Cursor cursor = db.query(mTableName, columns, selection, selectionArgs, groupBy, having, orderBy, limit);
        if (cursor != null) {
            cursor.moveToFirst();
        }

        return cursor;
    }

    /**
     * Execute query on current table
     *
     * @param optionalParams optional params to replace the SQLite question mark
     * @return the id of the inserted raw
     */
    public Cursor rawQuery(String sql, Object... optionalParams) {
        SQLiteDatabase readableDatabase = mDBHelper.getReadableDatabase();
        String[] params = new String[optionalParams.length];
        for (int i = 0; i < optionalParams.length; i++) {
            params[i] = optionalParams[i].toString();
        }
        return readableDatabase.rawQuery(sql, params);
    }


    /**
     * SQLite Insert or update query
     *
     * @return the id of the inserted raw or ROW_UPDATED if updated
     */
    public long insertOrUpdate(ContentValues contentValuesForInsert, ContentValues contentValuesForUpdate, String whereClause, String[] whereArgs) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        long rowId;
        try {
            db.beginTransaction();
            LPMobileLog.d(TAG, "insertOrUpdate started");

            rowId = insertOrUpdate(db, contentValuesForInsert, contentValuesForUpdate, whereClause, whereArgs);

            db.setTransactionSuccessful();
        } finally {
            try {
                db.endTransaction();
            } catch (Exception e) {
                LPMobileLog.w(TAG, "Error while ending transaction: ", e);
            }
        }
        return rowId;
    }

	/**
	 * SQLite Insert or update query
	 *
	 * @return the id of the inserted raw or ROW_UPDATED if updated
	 */
    private long insertOrUpdate(SQLiteDatabase db, ContentValues contentValuesForInsert, ContentValues contentValuesForUpdate, String whereClause, String[] whereArgs) {
        long rowId;
        Cursor c = db.rawQuery("SELECT * FROM " + mTableName + " WHERE " + whereClause, whereArgs);
        if (c.moveToFirst()) {
            c.close();
            LPMobileLog.d(TAG, "insertOrUpdate: update");
            db.update(mTableName, contentValuesForUpdate, whereClause, whereArgs);
			rowId = ROW_UPDATED; // Indicates this is an update
        } else {
            c.close();
            LPMobileLog.d(TAG, "insertOrUpdate: insert");
            rowId = db.insertWithOnConflict(mTableName, null, contentValuesForInsert, SQLiteDatabase.CONFLICT_IGNORE);
        }
        return rowId;
    }

    /**
     * Insert bulk of data to selected table
     *
     * @param sqliteCommands
     */
    public void runTransaction(List<SQLiteCommand> sqliteCommands) {

        SQLiteDatabase db = mDBHelper.getWritableDatabase();
		long rowId;

        try {
            db.beginTransaction();
            LPMobileLog.d(TAG, "Transaction started");
            for (SQLiteCommand command : sqliteCommands) {
                switch (command.getType()) {
                    case SQLiteCommand.CommandType.INSERT:
                        rowId = db.insertWithOnConflict(mTableName, null, command.getContentValues(), SQLiteDatabase.CONFLICT_IGNORE);
                        command.onInserted(rowId);
						break;
                    case SQLiteCommand.CommandType.INSERT_OR_UPDATE:
						rowId = insertOrUpdate(db, ((InsertOrUpdateSQLCommand) command).getContentValuesForInsert(), ((InsertOrUpdateSQLCommand) command).getContentValuesForUpdate() , ((InsertOrUpdateSQLCommand) command).getUpdateWhereClause(), ((InsertOrUpdateSQLCommand) command).getUpdateWhereArgs());
						if (rowId != ROW_UPDATED) { // If the row was inserted (not updated)
							command.onInserted(rowId);
						}
						break;
                    case SQLiteCommand.CommandType.UPDATE:
                        /*int updated =*/ db.update(mTableName, command.getContentValues(), ((UpdateSQLCommand) command).getUpdateWhereClause(), ((UpdateSQLCommand) command).getUpdateWhereArgs());
                       // LPMobileLog.d(TAG, ((UpdateSQLCommand) command).getUpdateWhereClause() + " : " +  ((UpdateSQLCommand) command).getUpdateWhereArgs() + " : " + updated);
                        break;
                }
            }
            db.setTransactionSuccessful();
        } finally {
            try{
                db.endTransaction();
            }catch (Exception e){
                LPMobileLog.w(TAG, "Error while ending transaction: ", e);
            }
        }
    }

    /**
     * SQLite update query
     *
     * @param values
     * @param whereClause
     * @param whereArgs
     * @return
     */
    public int update(ContentValues values, String whereClause, String[] whereArgs) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        return db.update(mTableName, values, whereClause, whereArgs);
    }

    /**
     * SQLite delete query
     *
     * @param whereClause
     * @param whereArgs
     * @return
     */
    public int removeAll(String whereClause, String[] whereArgs) {
        SQLiteDatabase db = mDBHelper.getWritableDatabase();
        return db.delete(mTableName, whereClause, whereArgs);
    }
}
