package com.liveperson.infra.handler;


import android.app.Notification;
import android.app.Notification.Builder;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.BitmapFactory;
import android.os.Build;
import android.support.annotation.RequiresApi;

import com.liveperson.infra.Infra;
import com.liveperson.infra.R;
import com.liveperson.infra.log.LPMobileLog;

/**
 * Created by Ilya Gazman on 11/26/2015.
 */
public class NotificationHandler {

    private static final String TAG = NotificationHandler.class.getSimpleName();
    private static final int NOTIFICATION_ID = 1212322;

    private final String mBrandId;
    private final String mTitle;
    private final String mMessage;
    private final Context mContext;
    private int mNotificationDefaults = Notification.DEFAULT_SOUND | Notification.DEFAULT_LIGHTS;
    private boolean mAutoCancel = true;

    private int mIconResourceId;
    private Class<?> mStartActivityClass;
    private PendingIntent mPendingIntent;

    public NotificationHandler(String brandId, Context context, Class<?> startActivityClass, String title, String message) {
        mContext = context;
        mBrandId = brandId;
        mStartActivityClass = startActivityClass;
        mTitle = title;
        mMessage = message;
    }

    public NotificationHandler(String brandId, Context context, Class<?> startActivityClass) {
        this(brandId, context, startActivityClass, null, null);
    }

    public NotificationHandler(Context context, String title){
        this(null, context, null, title, null);
    }

    /**
     * Set the icon resource for the notification
     *
     * @param iconResourceId
     * @return
     */
    public NotificationHandler setIconResourceId(int iconResourceId) {
        mIconResourceId = iconResourceId;
        return this;
    }

    /**
     * Set the pending intent for the notification
     *
     * @param pendingIntent
     * @return
     */
    public NotificationHandler setPendingIntent(PendingIntent pendingIntent) {
        mPendingIntent = pendingIntent;
        return this;
    }

    /**
     * Set auto cancel flag for the notification (default is <code>true</code>)
     *
     * @param autoCancel
     * @return
     */
    public NotificationHandler setAutoCancel(boolean autoCancel) {
        mAutoCancel = autoCancel;
        return this;
    }

    /**
     * Set the notification defaults. (default is: Notification.DEFAULT_SOUND | Notification.DEFAULT_LIGHTS)
     *
     * @param defaults
     * @return
     */
    public NotificationHandler setNotificationDefaults(int defaults) {
        mNotificationDefaults = defaults;
        return this;
    }

    /**
     * Hide notification
     */
    public static void hideNotification(Context context, String brandId) {
        ((NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE)).cancel(brandId, NOTIFICATION_ID);
    }

    /**
     * Show notification
     */
    public void showNotification() {
        LPMobileLog.i(TAG, "show notification, title = " + mTitle);

        Notification notification = createPushNotificationBuilder().build();
        getNotificationManager().notify(NOTIFICATION_ID, notification);
    }

    private Builder createPushNotificationBuilder() {
        String channelId = mContext.getString(R.string.push_notification_channel_id);
        String channelName = mContext.getString(R.string.push_notification_channel_name);

        Builder builder = createNotificationBuilder(channelId, channelName);
        builder.
                setContentTitle(mTitle).
                setAutoCancel(mAutoCancel).
                setDefaults(mNotificationDefaults).
                setSmallIcon(mIconResourceId).
                setContentText(mMessage);

        if (Build.VERSION.SDK_INT >= 21) {
            builder.
                    setCategory(Notification.CATEGORY_MESSAGE).
                    setPriority(Notification.PRIORITY_HIGH);
        }

        return builder;
    }

    public Builder createForegroundServiceNotificationBuilder() {
        String channelId = mContext.getString(R.string.foreground_service_notification_channel_id);
        String channelName = mContext.getString(R.string.foreground_service_notification_channel_name);

        Builder builder = createNotificationBuilder(channelId, channelName);
        builder = builder
                .setContentTitle(mTitle)
                .setSmallIcon(mIconResourceId)
                .setLargeIcon(BitmapFactory.decodeResource(mContext.getResources(), mIconResourceId))
                .setContentIntent(mPendingIntent)
                .setProgress(0, 0, true);

        return builder;
    }

    private Notification.Builder createNotificationBuilder(String channelId, String channelName) {
        Notification.Builder builder;
        if (Build.VERSION.SDK_INT < android.os.Build.VERSION_CODES.O) {
            builder = new Notification.Builder(mContext);
        } else {
            createNotificationChannel(channelId, channelName);
            builder = new Notification.Builder(mContext, channelId);
        }

        return builder;
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private void createNotificationChannel(String channelId, String channelName){
        NotificationChannel notificationChannel = new NotificationChannel(channelId, channelName, NotificationManager.IMPORTANCE_DEFAULT);
        getNotificationManager().createNotificationChannel(notificationChannel);
    }

    /**
     * Create a pending intent for the given brand
     *
     * @param brand
     * @return
     */
    private PendingIntent createPendingIntent(String brand) {
        LPMobileLog.d(TAG, "createPendingIntent with brand " + brand);
        Intent showIntent;
        if ((mStartActivityClass == null) || (mContext == null)) {

            LPMobileLog.e(TAG, "createPendingIntent: target class or context not available");
            return null;
        }

        showIntent = new Intent(mContext, mStartActivityClass);

        showIntent.putExtra(Infra.KEY_BRAND_ID, brand);
        return PendingIntent.getActivity(mContext, 10, showIntent, PendingIntent.FLAG_UPDATE_CURRENT);
    }

    /**
     * @return
     */
    private NotificationManager getNotificationManager() {
        return (NotificationManager) mContext.getSystemService(Context.NOTIFICATION_SERVICE);
    }
}
