package com.liveperson.infra.utils;

import android.content.ContentResolver;
import android.content.Context;
import android.database.Cursor;
import android.net.Uri;
import android.provider.OpenableColumns;
import android.text.TextUtils;
import android.webkit.MimeTypeMap;

import com.liveperson.infra.log.LPMobileLog;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

public class DocumentUtils {

    private static final String TAG = "DocumentUtils";

    private static final String DOCUMENTS_FOLDER = "documents/";
    private static final List<String> knownExtensions = Arrays.asList("pdf", "docx", "pptx", "xlsx");

    /**
     * calculate and return document size from URI
     * @param uri file uri
     * @param context application context
     * @return
     */
    public static int getDocumentSize(Uri uri, Context context) {

        String fileScheme = uri.getScheme();
        if (fileScheme != null && fileScheme.equals(ContentResolver.SCHEME_CONTENT)) {
            // The query, since it only applies to a single document, will only return
            // one row. There's no need to filter, sort, or select fields, since we want
            // all fields for one document.
            Cursor cursor = context.getContentResolver()
                    .query(uri, null, null, null, null, null);

            try {
                // moveToFirst() returns false if the cursor has 0 rows.  Very handy for
                // "if there's anything to look at, look at it" conditionals.
                if (cursor != null && cursor.moveToFirst()) {
                    int sizeIndex = cursor.getColumnIndex(OpenableColumns.SIZE);
                    String size;
                    if (!cursor.isNull(sizeIndex)) {
                        size = cursor.getString(sizeIndex);
                        return Integer.parseInt(size);
                    } else {
                        return 0;
                    }
                }
            } catch (Exception e) {
                LPMobileLog.e(TAG, "Failed to get document size", e);
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            }
        } else {
            try {
                File file = new File(uri.getPath());
                int size = (int) file.length();
                LPMobileLog.d(TAG, "File size: " + size);
                return size;
            } catch (Exception e) {
                LPMobileLog.e(TAG, "Failed to calculate file size", e);
            }
        }
        return 0;
    }

    /**
     * calculate and return document name from URI
     * @param uri
     * @param context
     * @return
     */
    public static String getDocumentName(Uri uri, Context context) {
        String fileScheme = uri.getScheme();
        if (fileScheme != null && fileScheme.equals(ContentResolver.SCHEME_CONTENT)) {
            Cursor cursor = context.getContentResolver()
                    .query(uri, null, null, null, null, null);
            try {
                if (cursor != null && cursor.moveToFirst()) {

                    // Note it's called "Display Name".  This is
                    // provider-specific, and might not necessarily be the file name.
                    String displayName = cursor.getString(
                            cursor.getColumnIndex(OpenableColumns.DISPLAY_NAME));
                    LPMobileLog.i(TAG, "File Name: " + displayName);
                    return displayName;
                }
            } finally {
                if (cursor != null) {
                    cursor.close();
                }
            } // end finally
        } else {
            try {
                File file = new File(uri.getPath());
                return file.getName();
            } catch (Exception e) {
                LPMobileLog.e(TAG, "Failed to retrieve file name", e);
            }
        }
        return null;
    }

    /**
     * Create copy of external file into app data folder and return new file path
     * @param uri File uri to be copied
     * @param context application context
     * @param brandId brandId as sub directory
     * @param documentByteArray actual document byte array
     * @return
     */
    public static String saveOriginalDocument(Uri uri, Context context, String brandId, byte[] documentByteArray) {
        String documentName = getDocumentName(uri, context);
        String filename = UUID.randomUUID().toString();

        String fileExtension = getDocumentExtension(uri, context, documentName);
        if (fileExtension != null) {
            filename = filename + "." + fileExtension;
        }

        LPMobileLog.d(TAG, "new file name: " + filename);
        return saveDocumentToDisk(context, filename, brandId, documentByteArray);
    }

    /**
     * Return extension based on file type if already does not exists.
     * @param uri
     * @param context
     * @param documentName doc name to retrieve extension from
     * @return
     */
    private static String getDocumentExtension(Uri uri, Context context, String documentName) {
        String documentNameArray[] = documentName.split("\\.");
        String extension = documentNameArray[documentNameArray.length-1];

        if (!TextUtils.isEmpty(extension) && knownExtensions.contains(extension.toLowerCase())) {
            return extension;
        } else if(!TextUtils.isEmpty(extension)) { // append return file extension based on file type
            String mimeType = context.getContentResolver().getType(uri);
            extension = MimeTypeMap.getSingleton().getExtensionFromMimeType(mimeType);
            return extension;
        }
        return null;
    }

    private static String saveDocumentToDisk(Context context, String fileName, String brandId, byte[] documentByteArray) {
        String path = "/" + DOCUMENTS_FOLDER + brandId;
        File filePath = FileUtils.getFilePath(context, path, fileName);
        if (filePath == null) {
            return null;
        }
        return saveDocumentToDisk(documentByteArray, filePath);
    }


    /**
     * Save the given byte array to disk for a given file
     *
     * @param documentByteArray document byte array
     * @param filePath file path where we want to same out file
     * @return path to saved file
     */
    private static String saveDocumentToDisk(byte[] documentByteArray, File filePath) {

        LPMobileLog.d(TAG, "saveDocumentToDisk: filePath: " + filePath.getAbsolutePath());

        FileOutputStream fos = null;
        try {
            fos = new FileOutputStream(filePath);
            fos.write(documentByteArray);
        } catch (FileNotFoundException e) {
            LPMobileLog.e(TAG, "saveDocumentToDisk: File not found", e);
            return null;
        } catch (IOException e) {
            LPMobileLog.e(TAG, "saveDocumentToDisk: IOException", e);
        } finally {
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                LPMobileLog.e(TAG, "saveDocumentToDisk: error closing file", e);
                return null;
            }
        }

        LPMobileLog.d(TAG, "saveDocumentToDisk: file absolute path: " + filePath.getAbsolutePath());
        return filePath.getAbsolutePath();
    }

}
