package com.liveperson.infra.network.socket;

import com.liveperson.infra.utils.UniqueID;

import java.util.ArrayList;

/**
 * Created by Ilya Gazman on 11/4/2015.
 *
 * The base request to be sent to SocketManager
 */
public abstract class BaseSocketRequest<T, REQUEST extends BaseSocketRequest> {

    private ArrayList<ResponseCallback<T>> mCallbacks = new ArrayList<>();
    private long mRequestId;

    public static long createRequestId(){
       return UniqueID.getUniqueID();
    }
	/**
	 * This constructor will generate and assign a unique ID to the requestId
	 */
    public BaseSocketRequest(){
		mRequestId = createRequestId();
    }

	/**
	 * This constructor will set the requestId with the given one. This is used for resending a message
	 * with the same request
	 * @param requestId
	 */
	public BaseSocketRequest(long requestId){
		mRequestId = requestId;
	}

    /**
     * @return the url for socket connection
     */
    protected abstract String getSocketUrl();

    /**
     * @return the data to be sent over the socket
     */
    protected abstract String getData();

    /**
     * @return the name of the request
     */
    protected abstract String getRequestName();

    /**
     * @return a response that will be mapped for this request over request id that we will get back from the server
     */
    protected abstract BaseResponseHandler<T, REQUEST> getResponseHandler();

    /**
     * return a unique id associated with this request
     */
    public long getRequestId() {
        return mRequestId;
    }

    /**
     * Sets a unique id to be associated
     * @param requestId
     * @return this class for builder pattern
     */
    public BaseSocketRequest setRequestId(long requestId) {
        mRequestId = requestId;
        return this;
    }

    /**
     * Adds a callback for this request
     * @param callback a callback for this request
     * @return this class for builder pattern
     */
    public REQUEST addCallback(ResponseCallback<T> callback) {
        mCallbacks.add(callback);
        //noinspection unchecked
        return (REQUEST) this;
    }

    /**
     * Removes a callback for this request
     * @param callback a callback for this request
     * @return this class for builder pattern
     */
    public REQUEST removeCallback(ResponseCallback<T> callback) {
        mCallbacks.remove(callback);
        //noinspection unchecked
        return (REQUEST) this;
    }

    /**
     notify the callback with error
     */
    void dispatchError() {
        for (ResponseCallback<T> callback : mCallbacks) {
            callback.onError();
        }
    }

    /**
     notify the callback with success
     */
    void dispatchSuccess(T data) {
        for (ResponseCallback<T> callback : mCallbacks) {
            callback.onSuccess(data);
        }
    }
}
