package com.liveperson.infra.log

import com.liveperson.infra.BuildConfig
import com.liveperson.infra.Infra
import com.liveperson.infra.log.LogLevel.*
import org.jetbrains.annotations.TestOnly

/**
 * Log helper.
 * Should be used only by the LivePerson SDK, not by the host app or others.
 */
object LPLog {

	private const val TAG = "LivePerson"

	private const val MAX_LOG_HISTORY = 100
	private const val START_OF_LOGS = "== Start of LivePerson Logs =="

	private var loggingDelegate: LoggingDelegate = AndroidLoggingDelegate()
	private val history = LoggingQueue(MAX_LOG_HISTORY)

	var loggingLevel = if (BuildConfig.DEBUG) VERBOSE else INFO
	// By default, INFO in Prod and VERBOSE in DEBUG

	var dataMaskingEnabled = !BuildConfig.DEBUG
	// By default, enable in Prod and disable in Debug

	init {
		history.add(LogLine(INFO, TAG, START_OF_LOGS, null))
	}

	/**
	 * Sets a custom backing logger. Used only for unit tests that cannot access the
	 * Android logger classes.
	 */
	@TestOnly
	fun setLoggingDelegate(loggingDelegate: LoggingDelegate) {
		LPLog.loggingDelegate = loggingDelegate
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun v(tag: String, message: String) {
		v(tag, message, null)
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	fun v(tag: String, flowTag: FlowTags, message: String) {
		v("$tag::${flowTag.apiName}", message, null)
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun v(tag: String, flowTag: FlowTags, message: String, exception: Throwable?) {
		v("$tag::${flowTag.apiName}", message, exception)
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun v(tag: String, message: String, exception: Throwable?) {
		history.add(LogLine(VERBOSE, tag, message, exception))
		if (VERBOSE <= loggingLevel) {
			loggingDelegate.v(TAG, "[[$tag]]  $message", exception)
		}
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun d(tag: String, message: String) {
		d(tag, message, null)
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	fun d(tag: String, flowTag: FlowTags, message: String) {
		d("$tag::${flowTag.apiName}", message, null)
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun d(tag: String, flowTag: FlowTags, message: String, exception: Throwable) {
		d("$tag::${flowTag.apiName}", message, exception)
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun d(tag: String, message: String, exception: Throwable?) {
		history.add(LogLine(DEBUG, tag, message, exception))
		if (DEBUG <= loggingLevel) {
			loggingDelegate.d(TAG, "[[$tag]]  $message", exception)
		}
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun i(tag: String, message: String) {
		i(tag, message, null)
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	fun i(tag: String, flowTag: FlowTags, message: String) {
		i("$tag::${flowTag.apiName}", message, null)
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun i(tag: String, flowTag: FlowTags, message: String, exception: Throwable) {
		i("$tag::${flowTag.apiName}", message, exception)
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun i(tag: String, message: String, exception: Throwable?) {
		history.add(LogLine(INFO, tag, message, exception))
		if (INFO <= loggingLevel) {
			loggingDelegate.i(TAG, "[[$tag]]  $message", exception)
		}
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun w(tag: String, message: String) {
		w(tag, message, null)
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	fun w(tag: String, flowTag: FlowTags, message: String) {
		w("$tag::${flowTag.apiName}", message, null)
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun w(tag: String, flowTag: FlowTags, message: String, exception: Throwable) {
		w("$tag::${flowTag.apiName}", message, exception)
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun w(tag: String, message: String, exception: Throwable?) {
		history.add(LogLine(WARNING, tag, message, exception))
		if (WARNING <= loggingLevel) {
			loggingDelegate.w(TAG, "[[$tag]]  $message", exception)
		}
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	fun e(tag: String, message: String) {
		e(tag, message, null)
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	fun e(tag: String, flowTag: FlowTags, message: String) {
		e("$tag::${flowTag.apiName}", message, null)
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun e(tag: String, flowTag: FlowTags, message: String, exception: Throwable) {
		e("$tag::${flowTag.apiName}", message, exception)
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	fun e(tag: String, message: String, exception: Throwable?) {
		history.add(LogLine(ERROR, tag, message, exception))
		if (ERROR <= loggingLevel) {
			loggingDelegate.e(TAG, "[[$tag]]  $message", exception)
		}

		if (Infra.instance.isInitialized) {
			Infra.instance.loggos.reportError()
		}
	}

	/**
	 * Masks potentially sensitive data, to be unreadable in the logs. Only modifies
	 * the input if `dataMaskingEnabled` is **true**.
	 *
	 * If input is modified, the following masking map is used:
	 * - null → null
	 * - "" → ""
	 * - all other cases → "********"
	 */
	fun mask(pii: Any?): String? {
		return if (dataMaskingEnabled) {
			when (pii?.toString()) {
				null -> null
				"" -> ""
				else -> "********"
			}
		} else {
			pii?.toString()
		}
	}

	/**
	 * Gets a snapshot of log history, as a List of LogLine objects, filtered by
	 * LogLevel.
	 *
	 * @param level The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a List<LogLine> containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshot(level: LogLevel): List<LogLine> {
		return history.getLogSnapshot(level)
	}

	/**
	 * Gets a snapshot of log history, as a List of Strings, filtered by LogLevel.
	 *
	 * @param level The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a List<String> containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshotStrings(level: LogLevel): List<String> {
		return history.getLogSnapshotStrings(level)
	}

	/**
	 * Gets a snapshot of log history, as a single String with each line separated by
	 * newline characters, filtered by LogLevel.
	 *
	 * @param level The highest LogLevel of lines you want included in the snapshot.
	 *
	 * @return a large String containing all log lines at an equal or lesser logging
	 * level than `filter`, and none of the lines at a greater logging level.
	 */
	fun getLogSnapshotStringBlock(level: LogLevel): String {
		return history.getLogSnapshotStringBlock(level)
	}

	/**
	 * Clears the in-memory log line cache.
	 */
	fun clearHistory() {
		history.clear()
		history.add(LogLine(INFO, TAG, START_OF_LOGS, null))
	}
}
