package com.liveperson.infra.network.socket;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.liveperson.infra.IDisposable;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.model.SocketConnectionParams;
import com.liveperson.infra.utils.DispatchQueue;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Created by ofira on 11/23/17.
 */

public class SocketHandler implements ISocketCallbacks, IDisposable {

    private static final String TAG = "SocketHandler";
    private static final int CLOSING_TIMEOUT = 5 * 1000;
    public static final int PERIODIC_PING_TIME = 20 * 1000;
    public static final int CERTIFICATE_ERROR = 1200;
    private ResponseMap mResponseMap;
    private DispatchQueue mRequestsQueue;
    private DispatchQueue mResponsesQueue;
    private ISocketWrapper mSocketWrapper;
    private SocketStateManager mSocketStateManager;
    private Runnable mClosingRunnable = null;

    /**
     * Creates new SocketHandler
     * @param responseMap map for responses
     */
    public SocketHandler(ResponseMap responseMap) {
        mRequestsQueue = new DispatchQueue(TAG + "_Requests");
        mResponsesQueue = new DispatchQueue(TAG + "_Responses");
        mSocketStateManager = new SocketStateManager();
        mResponseMap = responseMap;
    }

    /**
     * Handling the socket connection
     */
    void connect(final SocketConnectionParams connectionParams) {
        mRequestsQueue.postRunnable(() -> handleConnect(connectionParams));
    }

    /**
     * handle connection request
     * This method needs to be called from the dispatch queue in order to keep it sync
     */
    private void handleConnect(SocketConnectionParams connectionParams) {
        //Runs in the handler thread
        SocketState state = mSocketStateManager.getState();
        LPLog.INSTANCE.d(TAG, "handleConnect with state " + state + ". ");
        switch (state) {
             case OPEN:
                 disconnect();
                 break;
            case INIT:
            case ERROR:
            case CLOSED:
                openConnection(connectionParams);
                break;
            default:
                //We ignore (OPEN, CONNECTING) states
                break;
        }
    }

    /**
     * Send close frame on the socket
     */
    void disconnect() {
	    LPLog.INSTANCE.d(TAG, "disconnect");
        mRequestsQueue.postRunnable(() -> mSocketWrapper.disconnect());
    }

    /**
     * Get the socket state manager of this socket
     */
    SocketStateManager getSocketStateManager() {
        return mSocketStateManager;
    }

    /**
     * Open socket connection
     */
    private void openConnection(SocketConnectionParams connectionParams) {
        //Runs in the handler thread
        LPLog.INSTANCE.d(TAG, "openConnection");
        mSocketWrapper = new SocketWrapperOK(this);
        try {
            mSocketWrapper.connect(connectionParams);
        }catch (IllegalArgumentException ie){
            LPLog.INSTANCE.d(TAG, "Error: ", ie);
        }
    }


    /**
     * Send message over the socket
     *
     * @param message Message to send.
     */
    public void send(final String message) {
        mRequestsQueue.postRunnable(new SocketHandler.SendMessageRunnable(message));
    }

    /**
     * Handle the value object in the response handler and notify the request with success callback
     */
    private void handle(BaseResponseHandler baseResponseHandler, Object valueObject) {
        BaseSocketRequest request = baseResponseHandler.getRequest();
        if (request != null) {
            request.dispatchSuccess(valueObject);
        }
        boolean isHandled = baseResponseHandler.handle(valueObject);
        if (request != null && isHandled) {
            mResponseMap.onRequestHandled(request.getRequestId());
        }
    }

    @Override
    public void onStateChanged(SocketState newState) {
        LPLog.INSTANCE.d(TAG, "onStateChanged newState " + newState.name());
        mSocketStateManager.setState(newState);
        switch (newState) {
            case CLOSING:
                mClosingRunnable = () -> {
                    //if we are on state closing after CLOSING_TIMEOUT we move to CLOSE state
                    if (mSocketStateManager.getState() == SocketState.CLOSING) {
	                    LPLog.INSTANCE.d(TAG, "onStateChanged timeout expired on state CLOSING. force closing socket. ");
                        finalizeClosing();
                    }
                };
                mRequestsQueue.postRunnable(mClosingRunnable, CLOSING_TIMEOUT);
                break;

            case CLOSED:
                if (mClosingRunnable != null) {
                    mRequestsQueue.removeRunnable(mClosingRunnable);
                    mClosingRunnable = null;
                }
                //Add the final close to the end of the queue -
                // in order to finish handling all the pending request before closing
                mResponsesQueue.postRunnable(this::finalizeClosing);
                break;
            default:
                break;
        }
    }

    private void finalizeClosing() {
        mSocketStateManager.setState(SocketState.CLOSED);
        mResponseMap.onSocketClosed();
    }

    @Override
    public void onMessage(final String text) {
        LPLog.INSTANCE.d(TAG, "---------------------onMessage---------------------");
        LPLog.INSTANCE.d(TAG, "text " + text);

        mResponsesQueue.postRunnable(new Runnable() {
            @Override
            public void run() {
                parseIncomingData();
            }

            private void parseIncomingData() {
                JSONObject jsonObject = null;
                try {
                    jsonObject = new JSONObject(text);
                } catch (JSONException e1) {
                    LPLog.INSTANCE.e(TAG, "Error converting response to json object! should never happened!", e1);
                }
                if (jsonObject == null) {
                    return;
                }

                String messageType = jsonObject.optString("type", "AbstractResponse");
                int requestId = jsonObject.optInt("reqId", -1);

                //Getting the response from requests's map.
                //If it's not exists there - it'll try to find general response
                BaseResponseHandler baseResponseHandler = mResponseMap.getRequestIdHandler(messageType, requestId);
                //get the type that this response can handle.
                String expectedType = extractExpectedParsingType(baseResponseHandler);

                LPLog.INSTANCE.d(ResponseMap.RESPONSE_TAG, "extractExpectedParsingType expectedType = " + expectedType + " received messageType = " + messageType);

                //if the expected response type we got is not the same as the one we know
                //baseResponseHandler can handle - we are trying to get the right response handler by this type
                if (baseResponseHandler != null && !TextUtils.equals(messageType, expectedType)) {
                    baseResponseHandler = baseResponseHandler.getResponseByExpectedType(messageType);
                }

                if (baseResponseHandler == null) {
                    LPLog.INSTANCE.d(TAG, "Lost response:" + messageType + "(" + requestId + ")");
                    LPLog.INSTANCE.i(TAG, "Got response = " + messageType + ", no response handler");

                } else {
                    Object valueObject = null;
                    try {
                        LPLog.INSTANCE.i(TAG, "Got response = " + messageType + (baseResponseHandler.getRequest() != null ? " for request " + baseResponseHandler.getRequest().getRequestName() : ""));
                        valueObject = baseResponseHandler.parse(jsonObject);
                    } catch (Exception e1) {
                        LPLog.INSTANCE.e(TAG, "Error parsing response!", e1);
                    }

                    if (valueObject == null) {
                        BaseSocketRequest request = baseResponseHandler.getRequest();
                        if (request != null) {
                            request.dispatchError();
                        }
                    } else {
                        handle(baseResponseHandler, valueObject);
                    }
                }
            }

            @Nullable
            private String extractExpectedParsingType(BaseResponseHandler baseResponseHandler) {
                if (baseResponseHandler == null) {
                    return null;
                }
                return baseResponseHandler.getAPIResponseType();
            }
        });
    }


    @Override
    public void onDisconnected(String reason, int code) {
        mSocketStateManager.disconnected(reason, code);
    }

    /**
     * Instantly stops the socket(move to CLOSED state), and clear any observers of socket state
     */
    @Override
    public void dispose() {
        mRequestsQueue.cleanupQueue();
        mResponsesQueue.cleanupQueue();
        mRequestsQueue.postRunnable(() -> {
            LPLog.INSTANCE.i(TAG, "dispose " + TAG);
            if (mSocketWrapper != null) {
                mSocketStateManager.dispose();
                mRequestsQueue.dispose();
                mResponsesQueue.dispose();
                mRequestsQueue = null;
                mResponsesQueue = null;
                mSocketStateManager = null;
                mSocketWrapper = null;
            }
        });
    }


    private class SendMessageRunnable implements Runnable {
        private final String message;

        SendMessageRunnable(String message) {
            this.message = message;
        }

        @Override
        public void run() {
            SocketState state = mSocketStateManager.getState();
            if (state == SocketState.OPEN) {
                LPLog.INSTANCE.d(TAG, "Sending data: " + LPLog.INSTANCE.mask(message));
                mSocketWrapper.send(message);
            } else {
                LPLog.INSTANCE.w(TAG, "Ignoring message(" + mSocketStateManager.getState() + ") " + LPLog.INSTANCE.mask(message));
            }
        }
    }
}
