package com.liveperson.infra.statemachine;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.statemachine.interfaces.IEvent;
import com.liveperson.infra.statemachine.interfaces.IStateMachine;
import com.liveperson.infra.statemachine.interfaces.IStateMachineExecutor;
import com.liveperson.infra.utils.DispatchQueue;

/**
 * Created by shiranr on 13/03/2016.
 */
public class StateMachineExecutor implements IStateMachineExecutor {
    private final String logTag;
    private final IStateMachine stateMachine;
    private DispatchQueue mDispatchQueue;

    class RunnableWrapper implements Runnable {
        final IEvent event;

        RunnableWrapper(IEvent iEvent) {
            event = iEvent;
        }

        public void run() {
            LPLog.INSTANCE.d(logTag, stateMachine.activeState().toString() + " <-<- " + event.toString());
            event.accept(stateMachine.activeState());
        }
    }

    public StateMachineExecutor(String logTag, IStateMachine stateMachine) {
        this.logTag = logTag;
        this.stateMachine = stateMachine;
        mDispatchQueue = new DispatchQueue(logTag);
    }

    @Override
    public void shutdown() {
        mDispatchQueue.dispose();
    }
    public boolean isInitialized() {
        return !isThreadDead();
    }

    @Override
    public void post(IEvent event) {
        if (isThreadDead()){
            return;
        }
        mDispatchQueue.postRunnable(new RunnableWrapper(event));
    }

    @Override
    public Runnable schedule(IEvent event, long delay) {
        if (isThreadDead()){
            return null;
        }
        RunnableWrapper runnable = new RunnableWrapper(event);
        mDispatchQueue.postRunnable(runnable, delay);
        return runnable;
    }

    @Override
    public void cancel(Runnable delayedEvent) {
        if (isThreadDead()){
            return;
        }
        mDispatchQueue.removeRunnable(delayedEvent);
    }

    private boolean isThreadDead() {
        if (!mDispatchQueue.isAlive()){
            LPLog.INSTANCE.w(logTag, "State Machine DispatchQueue is dead.");
            return true;
        }
        return false;
    }

}
