package com.liveperson.infra.analytics

import android.content.Context
import android.os.Build
import android.text.TextUtils
import com.liveperson.infra.BuildConfig
import com.liveperson.infra.Infra
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.w
import com.liveperson.infra.loggos.LoggosUploader
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.managers.PreferenceManager.APP_ID_PREFERENCE_KEY
import com.liveperson.infra.utils.AnalyticsUtils
import com.liveperson.infra.utils.VersionUtils.getAppVersion
import com.liveperson.infra.utils.DeviceUtils.getOsName
import org.json.JSONObject
import java.util.*
import java.util.concurrent.LinkedBlockingQueue
import kotlin.collections.ArrayList
import kotlin.collections.HashMap

open class AnalyticsService {
    private lateinit var appContext: Context
    private lateinit var brandId: String
    private var appId: String = ""
    private var userProperties = HashMap<String, Any>()
    private var userEvents = LinkedBlockingQueue<AnalyticsEvent>()
    private var retryCount = 0
    private var isAnalyticsEnabled = true // Analytics is enabled by default

    private var sessionId: String = ""

    companion object {
        private const val TAG = "AnalyticsService"
        private const val EVENT_CACHE_SIZE = 10
        private const val CSDS_LOGGOS_DOMAIN_KEY = "loggos"

        private const val BRAND_ID = "accountID"
        private const val APP_ID = "hostAppName"
        private const val APP_VERSION = "hostAppVersion"
        private const val SDK_VERSION = "sdkVersion"
        private const val DEVICE_API = "deviceApi" // Not indexed in Kibana
        private const val DEVICE_LANGUAGE = "language"
        private const val DEVICE_SCRIPT = "deviceScript" // Not indexed in Kibana
        private const val DEVICE_REGION = "region"
        private const val DEVICE_TIME_ZONE = "geoip.timezone"
        private const val DEVICE_MAKE = "deviceFamily"
        private const val DEVICE_MODEL = "deviceModel"
        private const val DEVICE_OS = "deviceOS" // // Not indexed in Kibana
        private const val DEVICE_OS_VERSION = "deviceOSVersion"
        private const val LOG_TIME = "event_time"
        private const val SESSION_ID = "sessionId"
    }

    /**
     * Initialize the analytics service to track all user events during the time of interaction
     * with SDK.
     */
    fun init(applicationContext: Context, brandId: String, appId: String) {
        this.appContext = applicationContext
        this.brandId = brandId
        this.appId = appId
        setUserProperties()
    }

    /**
     * Add user event, appended with brandId and current time into list of events.
     */
    open fun logUserEvent(analyticsEvent: AnalyticsEvent?) {
        try {
            // Do not log event if analytics is disabled.
            if (!isAnalyticsEnabled || analyticsEvent == null) {
                d(TAG, "logUserEvent: Analytics is disabled. Do not log event.")
                return
            }
            // Append brandId and current time as event property
            val eventProperties = arrayOf(EventProperty(LOG_TIME, AnalyticsUtils.getStartTime()),
                    EventProperty(BRAND_ID, brandId)) + analyticsEvent.getEventProperties()

            d(TAG, "logUserEvent: Adding user event ${analyticsEvent.eventName} into the list.");
            userEvents.add(AnalyticsEvent(analyticsEvent.eventName, *eventProperties))
            d(TAG, "logUserEvent: Total events cached: " + userEvents.size)

            if (userEvents.size >= EVENT_CACHE_SIZE || shouldSendImmediately(analyticsEvent.eventName)) {
                sendAnalyticsDataToLoggos(false)
            }
        } catch (exception: Exception) {
            w(TAG, "logUserEvent: Failed to log user event: ", exception)
        }
    }


    /**
     * Create HashMap of user properties which is being sent along with user events.
     */
    private fun setUserProperties() {
        try {
            userProperties.put(APP_ID, getBrandAppId())
            userProperties.put(APP_VERSION, getAppVersion(appContext))
            userProperties.put(SDK_VERSION, BuildConfig.VERSION_NAME)
            userProperties.put(BRAND_ID, brandId)
            userProperties.put(DEVICE_API, Build.VERSION.SDK_INT)
            userProperties.put(DEVICE_LANGUAGE, Locale.getDefault().toString())
//            userProperties.put(DEVICE_SCRIPT, getDeviceScript()) // Revisit this
            userProperties.put(DEVICE_REGION, Locale.getDefault().country)
            userProperties.put(DEVICE_TIME_ZONE, TimeZone.getDefault().id)
            userProperties.put(DEVICE_MAKE, Build.MANUFACTURER)
            userProperties.put(DEVICE_MODEL, Build.MODEL)
            userProperties.put(DEVICE_OS, getOsName())
            userProperties.put(DEVICE_OS_VERSION, Build.VERSION.RELEASE)
            userProperties.put(SESSION_ID, getSessionId())
        } catch (e: Exception) {
            w(TAG, "setUserProperties: Exception while mapping user properties.", e)
        }
    }

    /**
     * Send analytics data report to Loggos as soon as user moves back from conversation
     * screen, logs out of sdk or we have reached cache limit. This report includes user properties as well as all events
     * collected during the time of user interaction.
     */
    fun sendAnalyticsDataToLoggos(shouldClearAllEvents: Boolean) {
        if (userProperties.isEmpty()) { // Could be empty because user is not in conversation screen
            setUserProperties()
        }

        if (isAnalyticsEnabled && (userProperties.isNotEmpty() || !userEvents.isEmpty())) {
            Infra.instance.loggos.uploadAnalyticsReport(userProperties, userEvents, object : LoggosUploader.Callback {
                override fun onSuccess(logId: String?) {
                    i(TAG, "sendAnalyticsDataToLoggos: onSuccess: Uploaded report to Loggos for session: $sessionId")
                    retryCount = 0
                    if (shouldClearAllEvents) {
                        userProperties.clear()
                        userEvents.clear()
                    } else {
                        var count = EVENT_CACHE_SIZE
                        // remove first 10 events from queue which we have already sent
                        while (userEvents.size > 0 && count > 0) {
                            userEvents.poll()
                            count--
                        }
                    }
                }

                override fun onError(messages: List<JSONObject>?, exception: Throwable?) {
                    w(TAG, "sendAnalyticsDataToLoggos: onError: Failed to upload report", exception)
                    if (exception?.message.equals("Loggos is not initialized.") && retryCount < 4) {
                        initLoggosAndRetry(shouldClearAllEvents)
                    } else {
                        retryCount = 0
                    }
                }
            })
        }
    }

    /**
     * If loggos service not initialized from messaging, we will try to do it here and
     * retry sending analytics data
     */
    private fun initLoggosAndRetry(shouldClearAllEvents: Boolean) {
        val loggosDomain = PreferenceManager.getInstance().getStringValue(CSDS_LOGGOS_DOMAIN_KEY, brandId, null)
        if (!TextUtils.isEmpty(loggosDomain)) {
            Infra.instance.loggos.init(loggosDomain, brandId, ArrayList())
            retryCount++
            sendAnalyticsDataToLoggos(shouldClearAllEvents)
        } else {
            d(TAG, "initLoggosAndRetry: Loggos domain not found. Cannot send analytics report.")
        }
    }

    /**
     * Set brand Id in case init is not yet called (mostly because SDK was in background)
     */
    fun setBrandId(brandId: String?) {
        if (brandId != null && brandId.isNotEmpty()) {
            this.brandId = brandId
        }
    }

    /**
     * Set host app context If init is not called yet (mostly because SDK was in background)
     */
    fun setAppContext(appContext: Context?) {
        if (appContext != null) {
            this.appContext = appContext
        }
    }

    /**
     * Set If analytics is enabled in site settings
     */
    fun setIsAnalyticsEnabled(isEnabled: Boolean) {
        isAnalyticsEnabled = isEnabled
    }

    /**
     * If logged event is one of the following types, send analytics report to loggos immediately
     * instead of caching event into 'userEvents' list.
     */
    private fun shouldSendImmediately(evenName: Any): Boolean {
        return (evenName == EventName.LivePerson.LOGOUT ||
                evenName == EventName.LivePerson.HANDLE_PUSH)
    }

    /**
     * Return brand's appId from shared preference If missing locally
     */
    private fun getBrandAppId(): String {
        if (appId.isEmpty() && brandId.isNotEmpty()) {
            d(TAG, "getBrandAppId: Missing appId, trying to get it from shared preference.")
            appId = PreferenceManager.getInstance().getStringValue(APP_ID_PREFERENCE_KEY, brandId, "")
        }
        return appId
    }

    /**
     * Generate unique sessionId to track user events for each session
     */
    private fun getSessionId(): String {
        if (sessionId.isEmpty()) {
            sessionId = UUID.randomUUID().toString()
        }
        return sessionId
    }
}