package com.liveperson.infra.configuration;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.support.annotation.BoolRes;
import android.support.annotation.ColorRes;
import android.support.annotation.DimenRes;
import android.support.annotation.IntegerRes;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;
import android.support.v4.content.res.ResourcesCompat;

import com.liveperson.infra.Infra;
import com.liveperson.infra.R;
import com.liveperson.infra.log.LPLog;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000002F;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000030;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000031;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000032;
import static com.liveperson.infra.errors.ErrorCode.ERR_00000033;
import static com.liveperson.infra.utils.Utils.getResources;

/**
 * Created by Perry on 08/03/2018.
 * Provides a configurable provider for RUNTIME configurations from XML resource files.
 * Be very careful when using this class. It's only used for test in MessagingTest app. Customers are using branding.xml to make configuration.
 * Please don't use this class unless need for test case. When add new resource, please make sure add it in runtime_configuration_keys.json
 */
public class Configuration {
    private static final String TAG = "Configuration";

    private static final String SHARED_PREFERENCES_FILE_NAME = "lp_runtime_config";
    private static final String RUNTIME_CONFIGURATION_KEYS = "runtimeConfigurationKeys";

    public static boolean getBoolean(@BoolRes int resId) throws Resources.NotFoundException {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        return sharedPreferences.getBoolean(String.valueOf(resId), context.getResources().getBoolean(resId));
    }

    public static float getDimension(@DimenRes int resId) {
        float result;
        Context context = Infra.instance.getApplicationContext();
        try {
            SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
            result = sharedPreferences.getFloat(String.valueOf(resId), context.getResources().getDimension(resId));
        } catch (Resources.NotFoundException e) {
            LPLog.INSTANCE.e(TAG, ERR_0000002F, "getDimension: ", e);
            result = (float) -1;
        }
        return result;
    }

    public static int getInteger(@IntegerRes int resId) throws Resources.NotFoundException {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        return sharedPreferences.getInt(String.valueOf(resId), context.getResources().getInteger(resId));
    }

    /**
     * This method is only used to get branding.xml String resources. Don't use it to get localized String.
     */
    public static String getString(@StringRes int resId) throws Resources.NotFoundException {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        return sharedPreferences.getString(String.valueOf(resId), context.getResources().getString(resId));
    }

    public static int getColor(@ColorRes int resId) {
        int result;
        Context context = Infra.instance.getApplicationContext();
        try {
            SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
            result = sharedPreferences.getInt(String.valueOf(resId), ResourcesCompat.getColor(context.getResources(), resId, null));
        } catch (Resources.NotFoundException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000030, "getColor: ", e);
            result = -1;
        }
        return result;
    }

    /**
     * Sets a custom string value
     *
     * @param resId       The ID of the custom value
     * @param stringValue The custom string value for the given ID
     */
    public static void set(int resId, String stringValue) {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        sharedPreferences.edit().putString(String.valueOf(resId), stringValue).apply();

        save(resId, stringValue);
    }

    /**
     * Sets a custom boolean value, used for toggle behaviours
     *
     * @param resId        The ID of the custom value
     * @param booleanValue The custom int value for the given ID
     */
    public static void set(int resId, boolean booleanValue) {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        sharedPreferences.edit().putBoolean(String.valueOf(resId), booleanValue).apply();

        save(resId, booleanValue);
    }

    /**
     * Sets a custom float value, used for dimensions
     *
     * @param resId      The ID of the custom value
     * @param floatValue The custom float value for the given ID
     */
    public static void set(int resId, float floatValue) {
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        sharedPreferences.edit().putFloat(String.valueOf(resId), floatValue).apply();

        save(resId, floatValue);
    }

    /**
     * Sets a custom int value, used for integers and colors
     *
     * @param resId    The ID of the custom value
     * @param intValue The custom int value for the given ID
     */
    public static void set(int resId, int intValue) {
        // For colors AND integers, it doesn't matter because the ID is different for each resource.
        Context context = Infra.instance.getApplicationContext();
        SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
        sharedPreferences.edit().putInt(String.valueOf(resId), intValue).apply();

        save(resId, intValue);
    }

    /**
     * Clear configurations.
     */
    public static void clearAll() {
        Context context = Infra.instance.getApplicationContext();
        context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE).edit().clear().apply();
    }

    private static void save(int resId, @Nullable Object resValue) {
        if (resValue == null) {
            Context context = Infra.instance.getApplicationContext();
            SharedPreferences sharedPreferences = context.getSharedPreferences(SHARED_PREFERENCES_FILE_NAME, Context.MODE_PRIVATE);
            sharedPreferences.edit().remove(String.valueOf(resId)).apply();
        }
    }

    /**
     * Using reflection to fetch all resources names of and their IDs from a given resource class.
     *
     * @param rInnerClass A class from teh generated R file
     */
    public static HashMap<Integer, String> scanXmlIdsAndNames(Class<?> rInnerClass) {
        @SuppressLint("UseSparseArrays") HashMap<Integer, String> fieldIdMap = new HashMap<>(); // Intentionally using 'UseSparseArrays', because later we'll use the ID from the key.
        if (rInnerClass == null) return fieldIdMap;

        try {
            Field[] fields = rInnerClass.getFields();
            Set<String> keys = getConfigurationKeySet();
            for (Field field : fields) {
                if (field == null) continue;
                int id = field.getInt(null);
                if (keys.contains(field.getName())) {
                    fieldIdMap.put(id, field.getName());
                }
            }
        } catch (final Exception e) {
            LPLog.INSTANCE.e(TAG, ERR_00000031, "Failed to generate key-value map.", e);
        }

        return fieldIdMap;
    }

    /**
     * @return A set of resources of runtime configuration
     */
    private static Set<String> getConfigurationKeySet() {
        Set<String> keySet = new HashSet<>();
        try {
            JSONObject configurationKeysJsonObject = new JSONObject(loadConfigurationKeysJsonString());
            JSONArray configurationKeysJsonArray = configurationKeysJsonObject.getJSONArray(RUNTIME_CONFIGURATION_KEYS);
            for (int i = 0; i < configurationKeysJsonArray.length(); i++) {
                keySet.add(configurationKeysJsonArray.getString(i));
            }
        } catch (JSONException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000032, "JSONException while parsing configurationKeys JSON.", e);
        }
        return keySet;
    }

    /**
     * @return Json String of configuration resources.
     */
    private static String loadConfigurationKeysJsonString() {
        String json = null;
        try {
            InputStream is = getResources().openRawResource(R.raw.runtime_configuration_keys);

            int size = is.available();
            byte[] buffer = new byte[size];

            //noinspection ResultOfMethodCallIgnored result number of bytes read is unnecessary
            is.read(buffer);
            is.close();

            json = new String(buffer, StandardCharsets.UTF_8);
        } catch (IOException e) {
            LPLog.INSTANCE.e(TAG, ERR_00000033, "IOException while loading configurationKeys from disk.", e);
        }
        return json;
    }
}
