package com.liveperson.infra.network.http.requests

import android.content.Context
import android.os.Build
import android.text.TextUtils
import com.liveperson.infra.BuildConfig
import com.liveperson.infra.ICallback
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.auth.LPAuthenticationType
import com.liveperson.infra.callbacks.AuthCallBack
import com.liveperson.infra.controller.DBEncryptionHelper
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.errors.ErrorCode.*
import com.liveperson.infra.log.LPLog.d
import com.liveperson.infra.log.LPLog.e
import com.liveperson.infra.log.LPLog.i
import com.liveperson.infra.log.LPLog.mask
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.model.errors.AuthError
import com.liveperson.infra.model.types.AuthFailureReason
import com.liveperson.infra.network.http.HttpHandler
import com.liveperson.infra.network.http.body.HttpRequestBody
import com.liveperson.infra.network.http.body.LPJSONObjectBody
import com.liveperson.infra.network.http.request.HttpPostRequest
import com.liveperson.infra.preferences.AuthPreferences
import com.liveperson.infra.utils.EncryptionVersion
import com.liveperson.infra.utils.TokenUtils
import org.json.JSONObject
import java.net.UnknownHostException
import javax.net.ssl.SSLPeerUnverifiedException

class AuthRequest(private val applicationContext: Context, private val brandId: String, private val idpDomain: String, private val lpAuthenticationParams: LPAuthenticationParams?,
                  private val hostVersion: String?, private val certificates: List<String?>?, private val connectorId: String?, private val authCallBack: AuthCallBack?) {

    companion object {
        private const  val TAG = "AuthRequest"
        private const val IDP_REQUEST_TIMEOUT = 30000
        private const val NETWORK_READ_TIMEOUT_ERROR_CODE = 598
        private const val NETWORK_CONNECT_TIMEOUT_ERROR_CODE = 599
        private const val USER_EXPIRED_ERROR = "2001"

        private const val SIGNUP = "signup"
        private const val AUTHENTICATE = "authenticate"

        private const val DEFAULT_REDIRECT_URI = "https://liveperson.net"
        private const val IDP_SIGNUP_URL = "https://%s/api/account/%s/%s?v=1.0"
        private const val IDP_AUTH_URL = "https://%s/api/account/%s/app/default/%s?v=2.0"
        private const val IDP_ANONYMOUS_URL = "https://%s/api/account/%s/anonymous/authorize"
        private const val IDP_UN_AUTH_URL = "https://%s/api/account/%s/app/%s/%s?v=3.0"
    }

    private var shouldCancelAuth: Boolean = false
    private var isAnonymousRequest: Boolean = false

    /**
     * Authenticate against IDP to extract Liveperson token/JWT to make authenticated requests to other LP services.
     */
    fun authenticate() {
        var authType: LPAuthenticationType? = LPAuthenticationType.UN_AUTH
        shouldCancelAuth = false
        if (lpAuthenticationParams != null) {
            authType = lpAuthenticationParams.authType
        }
        when (authType) {
            LPAuthenticationType.AUTH -> {
                //use idp version #2
            }
            LPAuthenticationType.UN_AUTH -> {
                isAnonymousRequest = true
            }
            LPAuthenticationType.SIGN_UP -> {
                //use sign up request = idp version #1
                if (BuildConfig.DEBUG) {
                    LPAuthenticationParams.printSignupDeprecationNotice()
                }
            }
            else -> {
                e(TAG, ERR_0000013B, "authenticate: Unknown authentication type found: $authType.")
                sendErrorCallback(Exception("authenticate: Unknown authentication type found: $authType."), AuthFailureReason.AUTH_INVALID)
                return
            }
        }
        i(TAG, "authenticate: Send request to authenticate. Consumer type: $authType")
        sendGeneralRequest(getHttpPostRequest(authType))
    }

    /**
     * Create and return Http post request object based on authentication type provided
     */
    private fun getHttpPostRequest(authType: LPAuthenticationType?): HttpPostRequest? {

        val httpPostRequest = HttpPostRequest(generateIdpRequestUrl(authType))
        val jsonBody = getRequestBody(authType)

        if (jsonBody != null) {
            val body: HttpRequestBody = LPJSONObjectBody(jsonBody)
            httpPostRequest.setBody(body)
        } else if (authType == LPAuthenticationType.AUTH) { // If failed to set body for AUTH type, terminate
            return null
        }

        httpPostRequest.callback = object : ICallback<String?, Exception?> {
            override fun onSuccess(idpResponse: String?) {
                d(TAG, "onSuccess " + mask(idpResponse))
                if (!TextUtils.isEmpty(idpResponse)) {
                    try {
                        val idpJson = JSONObject(idpResponse)
                        var token: String? = null
                        when (authType) {
                            LPAuthenticationType.AUTH -> {
                                token = idpJson.getString("token")
                            }
                            LPAuthenticationType.UN_AUTH -> {
                                token = idpJson.getString("token")
                                if (isAnonymousRequest) {
                                    isAnonymousRequest = false
                                    setUnAuthToken(token)
                                    sendGeneralRequest(getHttpPostRequest(authType))
                                    return
                                }
                            }
                            else -> {
                                // Default is SignUp
                                token = idpJson.getString("jwt")
                            }
                        }
                        handleSuccess(token)
                    } catch (exception: Exception) {
                        e(TAG, ERR_0000013C, "getHttpPostRequest: Failed to parse response: ", exception)
                        isAnonymousRequest = false
                        sendErrorCallback(Exception("getHttpPostRequest: Failed to parse response"), AuthFailureReason.CLIENT)
                    }
                }
            }

            override fun onError(exception: Exception?) {
                d(TAG, "Error: idp url = " + httpPostRequest.url + ". Exception ", exception)
                if (exception != null && authType == LPAuthenticationType.UN_AUTH) {
                    isAnonymousRequest = false
                    val message = exception.message
                    if (!TextUtils.isEmpty(message) && message!!.contains(USER_EXPIRED_ERROR) // The following check is super safety -
                            // we can get 2001- 'USER EXPIRED' error only when trying to refresh the current non auth token, so we must have it
                            && !TextUtils.isEmpty(getUnAuthToken())) {
                        setUnAuthToken("")
                        sendErrorCallback(exception, AuthFailureReason.USER_EXPIRED)
                        return
                    }
                }
                sendErrorCallback(exception, null)
            }
        }
        return httpPostRequest
    }

    /**
     * Return request body of type json object as per authentication type
     */
    private fun getRequestBody(authType: LPAuthenticationType?): JSONObject? {
        val jsonBody = JSONObject()
        try {
            when (authType) {
                LPAuthenticationType.AUTH -> {
                    val authKey: String? = lpAuthenticationParams?.authKey
                    val authJwt: String? = lpAuthenticationParams?.hostAppJWT

                    if (!TextUtils.isEmpty(authKey)) { // Auth code flow
                        jsonBody.put("code", authKey)
                        if (TextUtils.isEmpty(lpAuthenticationParams?.hostAppRedirectUri)) {
                            jsonBody.put("redirect_uri", DEFAULT_REDIRECT_URI)
                        } else {
                            jsonBody.put("redirect_uri", lpAuthenticationParams?.hostAppRedirectUri)
                        }
                    } else if (!TextUtils.isEmpty(authJwt)) { // Auth JWT/Implicit flow
                        jsonBody.put("id_token", authJwt)
                    } else {
                        sendErrorCallback(Exception("getRequestBody: Failed to authenticate. No JWT nor authKey was provided"), AuthFailureReason.AUTH_INVALID)
                        return null
                    }
                }
                LPAuthenticationType.UN_AUTH -> {
                    val unAuthToken: String? = getUnAuthToken()
                    if (!TextUtils.isEmpty(unAuthToken)) {
                        jsonBody.put("id_token", unAuthToken)
                    } else {
                        return null
                    }
                }
                else -> {
                    // Default is SignUp
                    return null
                }
            }
        } catch (exception: Exception) {
            e(TAG, ERR_0000013D, "getRequestBody: Failed to build request body for request of type: $authType. ", exception)
            sendErrorCallback(Exception("getRequestBody: Failed to build request body for request of type: $authType"), AuthFailureReason.CLIENT)
            return null
        }
        return jsonBody
    }

    /**
     * Generate IDP request URL based on type of authentication requested.
     */
    private fun generateIdpRequestUrl(authType: LPAuthenticationType?): String? {
        try {
            return when (authType) {
                LPAuthenticationType.AUTH -> {
                    String.format(IDP_AUTH_URL, idpDomain, brandId, AUTHENTICATE)
                }
                LPAuthenticationType.UN_AUTH -> {
                    if (isAnonymousRequest)
                        String.format(IDP_ANONYMOUS_URL, idpDomain, brandId)
                    else
                        String.format(IDP_UN_AUTH_URL, idpDomain, brandId, connectorId, AUTHENTICATE)
                }
                else -> {
                    // Default is SignUp
                    String.format(IDP_SIGNUP_URL, idpDomain, brandId, SIGNUP)
                }
            }
        } catch (error: Exception) {
            e(TAG, ERR_0000013E, "generateIdpRequestUrl: Failed to generate IDP request URL. ", error)
            sendErrorCallback(Exception("generateIdpRequestUrl: Failed to generate IDP request URL. $error"), AuthFailureReason.CLIENT)
            return null
        }
    }

    /**
     * Send Http post request by adding certificate pinning keys and headers
     */
    private fun sendGeneralRequest(httpPostRequest: HttpPostRequest?) {
        if (httpPostRequest != null && !shouldCancelAuth) {
            //Setting the certificate pinning
            httpPostRequest.certificatePinningKeys = certificates
            d(TAG, "sendGeneralRequest: IDP request url : " + httpPostRequest.url)
            httpPostRequest.timeout = IDP_REQUEST_TIMEOUT
            // Add headers
            addHeaders(httpPostRequest)
            // Execute the request
            HttpHandler.execute(httpPostRequest)
        }
    }

    /**
     * Add request headers
     */
    private fun addHeaders(httpPostRequest: HttpPostRequest) {
        httpPostRequest.addHeader("sdkVersion", hostVersion)
        httpPostRequest.addHeader("platform", "Android")
        httpPostRequest.addHeader("platformVer", Build.VERSION.SDK_INT.toString())
        httpPostRequest.addHeader("device", Build.MODEL)
        httpPostRequest.addHeader("applicationId", brandId.replace("\n", ""))
    }

    /**
     * Cancel authentication request. This should be called If we got logout or shutdown while authenticating.
     */
    fun cancelAuth() {
        shouldCancelAuth = true
    }

    private fun handleSuccess(token: String?) {
        if (!shouldCancelAuth) {
            val consumerId = TokenUtils.getConsumerUserId(token)
            d(TAG, "handleSuccess: Extracted consumerId: " + mask(consumerId))
            authCallBack?.onAuthSuccess(Consumer(lpAuthenticationParams, brandId, consumerId, token))
        }
    }

    /**
     * Return error as an object of type AuthError via callback
     */
    private fun sendErrorCallback(exception: java.lang.Exception?, authFailureReason: AuthFailureReason?) {
        e(TAG, ERR_0000013F, "sendErrorCallback: Sending error callback. (AuthFailureReason: $authFailureReason)", exception)
        try {
            var failureReason: AuthFailureReason? = authFailureReason
            var statusCode: Int? = null
            if (exception != null) {
                statusCode = extractStatusCode(exception.message)
            }

            // If we don't know failure reason, try to determine it based on status code or exception type
            if (authFailureReason == null) {
                if (exception is SSLPeerUnverifiedException) {
                    failureReason = AuthFailureReason.INVALID_CERTIFICATE
                } else if (exception is UnknownHostException) {
                    failureReason = AuthFailureReason.CLIENT
                } else if (statusCode == NETWORK_READ_TIMEOUT_ERROR_CODE || statusCode == NETWORK_CONNECT_TIMEOUT_ERROR_CODE) {
                    failureReason = AuthFailureReason.NETWORK
                } else if (statusCode != null && (statusCode in 500..596)) {
                    failureReason = AuthFailureReason.IDP_FAILURE
                } else {
                    failureReason = AuthFailureReason.UNKNOWN
                }
            }
            authCallBack?.onAuthFailed(AuthError(failureReason, exception, statusCode))
        } catch (error: Exception) {
            e(TAG, ERR_00000140, "sendErrorCallback: Failed to send error callback: ", error)
            authCallBack?.onAuthFailed(AuthError(AuthFailureReason.UNKNOWN, error, null))
        }
    }

    /**
     * Parse Exception message and extract response code from it.
     * This is really a hacky way because of the error message format we compose in HttpRequest class
     */
    private fun extractStatusCode(message: String?): Int? {
        return try {
            val subString = message?.substringAfter("response code: ")
            val code = subString?.substringBefore(' ')
            Integer.parseInt(code!!)
        } catch (error: Exception) {
            i(TAG, "extractResponseCode: Failed to parse Exception message: $error")
            null
        }
    }

    /**
     * Encrypt and store un auth token into shared preferences
     */
    private fun setUnAuthToken(token: String?) {
        if (token != null) {
            val encryptedToken = DBEncryptionHelper.encrypt(EncryptionVersion.VERSION_1, token)
            AuthPreferences.getInstance(applicationContext).setUnAuthToken(brandId, encryptedToken)
        }
    }

    /**
     * return decrypted un auth token from shared preferences
     */
    private fun getUnAuthToken(): String? {
        val encryptedToken = AuthPreferences.getInstance(applicationContext).getUnAuthToken(brandId, null)
        return DBEncryptionHelper.decrypt(EncryptionVersion.VERSION_1, encryptedToken)
    }

}