/* Copyright (c) LivePerson, Inc. */
package com.liveperson.infra.utils;

import android.support.annotation.Nullable;
import android.text.TextUtils;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.TimeZone;

/**
 * Created by shiranr on 05/06/2016.
 */
public class DateUtils {

    private static Calendar mCalendar = Calendar.getInstance();

    /**
     * @return true if the supplied when is yesterday else false
     */
    public static boolean isYesterday(long when) {

        //Set and get calendar from the message timestamp
        mCalendar.setTimeInMillis(when);

        int msgYear = mCalendar.get(Calendar.YEAR);
        int msgMonth = mCalendar.get(Calendar.MONTH);
        int msgMonthDay = mCalendar.get(Calendar.DAY_OF_MONTH);

        //Set and get calendar from yesterday timestamp
        mCalendar.setTimeInMillis(System.currentTimeMillis());
        mCalendar.add(Calendar.DATE, -1);
        int yesterdayYear = mCalendar.get(Calendar.YEAR);
        int yesterdayMonth = mCalendar.get(Calendar.MONTH);
        int yesterdayMonthDay = mCalendar.get(Calendar.DAY_OF_MONTH);

        return (msgYear == yesterdayYear) &&
                (msgMonth == yesterdayMonth) &&
                (msgMonthDay == yesterdayMonthDay);
    }

    /**
     * @param when
     * @return true if a given time was in the past 24 hours
     */
    public static boolean isInTheLast24hours(long when) {
        return (System.currentTimeMillis() - android.text.format.DateUtils.DAY_IN_MILLIS < when);
    }


    /**
     * @return true if the supplied when is tomorrow else false
     */
    public static boolean isTomorrow(long when) {

        //Set and get calendar from the message timestamp
        mCalendar.setTimeInMillis(when);

        int msgYear = mCalendar.get(Calendar.YEAR);
        int msgMonth = mCalendar.get(Calendar.MONTH);
        int msgMonthDay = mCalendar.get(Calendar.DAY_OF_MONTH);

        //Set and get calendar from tomorrow timestamp
        mCalendar.setTimeInMillis(System.currentTimeMillis());
        mCalendar.add(Calendar.DATE, +1);
        int tomorrowYear = mCalendar.get(Calendar.YEAR);
        int tomorrowMonth = mCalendar.get(Calendar.MONTH);
        int tomorrowMonthDay = mCalendar.get(Calendar.DAY_OF_MONTH);

        return (msgYear == tomorrowYear) &&
                (msgMonth == tomorrowMonth) &&
                (msgMonthDay == tomorrowMonthDay);
    }


    public static long getBeginningOfDayTime(long timeStamp) {
        mCalendar.setTimeInMillis(timeStamp);
        mCalendar.set(Calendar.HOUR_OF_DAY, 0);
        mCalendar.set(Calendar.MINUTE, 1);
        mCalendar.set(Calendar.SECOND, 0);
        mCalendar.set(Calendar.MILLISECOND, 0);
        return mCalendar.getTimeInMillis();
    }


    public static boolean isToday(long timeStamp) {
        return android.text.format.DateUtils.isToday(timeStamp);
    }

    /**
     * Get the time formatted according to device's locale.
     * @param customTimeFormat in case of specific time format. in case null - will use the default.
     * @param defaultTimeFormatLength - default time length in case customTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param timeToFormat the time to format.
     * @return formatted time.
     */
    public static String getFormattedTime(@Nullable  String customTimeFormat, int defaultTimeFormatLength, long timeToFormat) {
        DateFormat format = getTimeFormat(customTimeFormat, defaultTimeFormatLength);
        return format.format(new Date(timeToFormat));
    }

    /**
     * Get the time formatted according to device's locale.
     * @param customTimeFormat in case of specific time format. in case null - will use the default.
     * @param defaultTimeFormatLength - default time length in case customTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param timeToFormat the time to format.
     * @param customTimeZone format the time with specific time zone. can be null.
     * @return formatted time.
     */
    public static String getFormattedTime(@Nullable String customTimeFormat, int defaultTimeFormatLength, long timeToFormat, @Nullable String customTimeZone) {
        DateFormat format = getTimeFormat(customTimeFormat, defaultTimeFormatLength);
        if (!TextUtils.isEmpty(customTimeZone)){
            format.setTimeZone(TimeZone.getTimeZone(customTimeZone));
        }
        return format.format(new Date(timeToFormat));
    }


    /**
     * Get the formatted time.
     * @param customTimeFormat in case of specific time format. in case null - will use the default.
     * @param timeFormatLength - default time length in case customTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @return the formatter
     */
    private static DateFormat getTimeFormat(String customTimeFormat, int timeFormatLength) {
        DateFormat format;
        Locale locale = LocaleUtils.getInstance().getLocale();

        if (!TextUtils.isEmpty(customTimeFormat)){
            format = new SimpleDateFormat(customTimeFormat,locale);
        }else{
            format = DateFormat.getTimeInstance(timeFormatLength, locale);
        }
        return format;
    }

    /**
     * Get the date formatted according to device's locale.
     * @param customDateFormat in case of specific date format. in case null - will use the default.
     * @param dateFormatLength - default date length in case customDateFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param dateToFormat the date to format.
     * @return formatted date.
     */
    public static String getFormattedDate(String customDateFormat, int dateFormatLength, long dateToFormat) {
        DateFormat format;
        Locale locale = LocaleUtils.getInstance().getLocale();

        if (!TextUtils.isEmpty(customDateFormat)){
            format = new SimpleDateFormat(customDateFormat,locale);
        }else{
            format = DateFormat.getDateInstance(dateFormatLength, locale);
        }

        return format.format(new Date(dateToFormat));
    }

    /**
     * Get the date an time formatted according to device's locale.
     * @param customDateTimeFormat in case of specific date & time format. in case null - will use the default.
     * @param dateFormatLength - default date length in case customDateTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param timeFormatLength - default time length in case customDateTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param dateTimeToFormat the date and time to format.
     * @return formatted date and time.
     */
    public static String getFormattedDateAndTime(String customDateTimeFormat,int dateFormatLength, int timeFormatLength, long dateTimeToFormat) {
        DateFormat format = getDateFormat(customDateTimeFormat, dateFormatLength, timeFormatLength);
        return format.format(new Date(dateTimeToFormat));
    }

    /**
     * Get the formatted date.
     * @param customDateTimeFormat in case of specific date&time format. in case null - will use the default.
     * @param dateFormatLength - default date length in case customDateTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @param timeFormatLength - default time length in case customDateTimeFormat is null, one of : DateFormat.SHORT | DateFormat.MEDIUM | DateFormat.LONG | DateFormat.FULL
     * @return the formatter
     */
    public static DateFormat getDateFormat(String customDateTimeFormat, int dateFormatLength, int timeFormatLength) {
        DateFormat format;
        Locale locale = LocaleUtils.getInstance().getLocale();

        if (!TextUtils.isEmpty(customDateTimeFormat)){
            format = new SimpleDateFormat(customDateTimeFormat,locale);
        }else{
            format = DateFormat.getDateTimeInstance(dateFormatLength, timeFormatLength, locale);
        }
        return format;
    }

    /**
     * Get a duration string mm:ss from milliseconds
     *
     * @param durationInMilliseconds Time in milliseconds.
     */
	public static String getDurationString(long durationInMilliseconds) {
		Date date = new Date(durationInMilliseconds);
		DateFormat formatter = new SimpleDateFormat("mm:ss", Locale.getDefault());
		return formatter.format(date);
	}
}
