package com.liveperson.infra;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.utils.LocalBroadcast;

import static com.liveperson.infra.errors.ErrorCode.*;


/**
 * Created by Ofir Antebi on 12/9/2015.
 * Service that keep tracking the internet connection
 * and informs the observers when there is a change
 * <p/>
 * Note: In order to be informed about network state change
 * there is a need to register for local broadcasts
 * BROADCAST_INTERNET_CONNECTION_CONNECTED OR BROADCAST_INTERNET_CONNECTION_DISCONNECTED
 */
public class InternetConnectionService implements ShutDown {

    private static final String TAG = "InternetConnectionService";
    public static final String BROADCAST_INTERNET_CONNECTION_CONNECTED = "CONNECTION_CONNECTED";
    public static final String BROADCAST_INTERNET_CONNECTION_DISCONNECTED = "CONNECTION_DISCONNECTED";
    private ConnectionReceiver mConnectionReceiver;

    /**
     * Package private
     * Only the infra controller can create it
     */
    InternetConnectionService() {
        registeredReceiver();
    }

    public void registeredReceiver() {
        if (mConnectionReceiver == null){
            mConnectionReceiver = new ConnectionReceiver();
            LPLog.INSTANCE.d(TAG, "creating new receiver");
        }
        mConnectionReceiver.register(Infra.instance.getApplicationContext());
    }
    /**
     * Unregister the receiver from being informed about network changes
     */
    public void unRegisteredReceiver() {
        if (mConnectionReceiver != null){
            LPLog.INSTANCE.d(TAG, "un-registering the receiver");
            try {
                Infra.instance.getApplicationContext().unregisterReceiver(mConnectionReceiver);
            } catch (IllegalArgumentException e) {
                LPLog.INSTANCE.e(TAG, ERR_00000010, "Failed to un-register connection receiver. Reason: ", e);
            }
            mConnectionReceiver = null;
        }
    }

    /**
     * Check if there is available internet connection
     *
     * @return if network is available
     */
    public static boolean isNetworkAvailable() {

        Context applicationContext = Infra.instance.getApplicationContext();
        if (applicationContext != null){
            ConnectivityManager connectivityManager = (ConnectivityManager) applicationContext.getSystemService(Context.CONNECTIVITY_SERVICE);
            NetworkInfo networkInfo = connectivityManager.getActiveNetworkInfo();
            boolean b = networkInfo != null && networkInfo.isConnectedOrConnecting();
            LPLog.INSTANCE.d(TAG, "isNetworkAvailable: isConnected = " + b );
            LPLog.INSTANCE.d(TAG, "isNetworkAvailable: networkInfo = " + networkInfo );
            return b;

        }
        LPLog.INSTANCE.d(TAG, "isNetworkAvailable: app context is null!" );
        return false;
    }

    @Override
    public void shutDown() {
        unRegisteredReceiver();
    }


    /**
     * The receiver that gets the broadcasts from the system
     */
    private class ConnectionReceiver extends BroadcastReceiver {

        private final IntentFilter intentFilter;

        private boolean lastConnectionState;

        public ConnectionReceiver(){
            intentFilter = new IntentFilter();
            intentFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
        }

        public void register(Context context) {
            Intent currentStatus = context.registerReceiver(this, intentFilter);
            lastConnectionState = isNetworkAvailable();
            onReceive(context, currentStatus);
            //sendUpdateIntent(lastConnectionState);
            LPLog.INSTANCE.d(TAG, "RegisteredReceiver, currentStatus = " + currentStatus);
            LPLog.INSTANCE.d(TAG, "RegisteredReceiver, lastConnectionState = " + lastConnectionState);
        }

        @Override
        public void onReceive(Context context, Intent intent) {
            if (intent != null) {
                String action = intent.getAction();
                if (ConnectivityManager.CONNECTIVITY_ACTION.equals(action)) {
                    boolean isConnected = isNetworkAvailable();
                    LPLog.INSTANCE.d(TAG, "isConnected " + isConnected);
                    LPLog.INSTANCE.d(TAG, "lastConnectionState " + lastConnectionState);
                    if (lastConnectionState != isConnected) {
                        lastConnectionState = isConnected;
                        sendUpdateIntent(isConnected);
                        LPLog.INSTANCE.i(TAG, "Getting broadcast with action " + intent.getAction() +", connected = "+ lastConnectionState);
                    }
                }
            }
        }

        private void sendUpdateIntent(boolean isConnected) {
            String intentAction;
            if (isConnected) {
                intentAction = BROADCAST_INTERNET_CONNECTION_CONNECTED;
            } else {
                intentAction = BROADCAST_INTERNET_CONNECTION_DISCONNECTED;
            }
            LocalBroadcast.sendBroadcast(intentAction);
        }

    }
}
