package com.liveperson.infra.eventmanager

import android.content.Context
import com.liveperson.infra.ICallback
import com.liveperson.infra.callbacks.AuthStateSubscription
import com.liveperson.infra.errors.ErrorCode
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.managers.ConsumerManager
import com.liveperson.infra.managers.PreferenceManager
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.network.http.requests.EventManagerRequest
import com.liveperson.infra.preferences.EventManagerPreferences
import org.json.JSONObject
import java.lang.Exception

/**
 * Service to track Proactive/C2M message events to be logged into Event Manager.
 */
open class EventManagerService {
    private lateinit var brandId: String
    private var conversationId: String? = ""
    private var appId: String = ""
    private var domain: String? = ""

    private var isInitialized: Boolean = false
    private var lpToken: String = ""
    private var certificatePinningKeys: MutableList<String>? = null

    private var eventManagerPreferences = EventManagerPreferences()

    companion object {
        private const val TAG = "EventManagerService"

        private const val CHANNEL = "inapp"
        private const val VERSION = "1.0"
        private const val REPORTER = "mobile_sdk_android"

        private const val KEY_EVENT_TYPE = "eventType"
        private const val KEY_TIMESTAMP = "timestamp"
        private const val KEY_ACCOUNT_ID = "accountId"
        private const val KEY_TRANSACTION_ID = "transactionId"
        private const val KEY_OUTBOUND_NUMBER = "outboundNumber"
        private const val KEY_CHANNEL = "channel"
        private const val KEY_VERSION = "v"
        private const val KEY_REPORTER = "reporter"
        private const val KEY_VALUE = "value"
        private const val KEY_CONVERSATION_ID = "conversationId"
        private const val KEY_ERROR_CODE = "errCode"
        private const val KEY_ERROR_MESSAGE = "errMsg"
    }

    /**
     * Initialize the event manager service to track 3 events during the time of interaction
     * with SDK.
     */
    fun init(applicationContext: Context, brandId: String, appId: String) {
        this.brandId = brandId
        this.appId = appId

        eventManagerPreferences.init(applicationContext)
        isInitialized = true
    }

    /**
     * Store an event of type EventManager in shared log preferences.
     * If we have got 'conversation' event, send it to event manager immediately.
     */
    open fun logEvent(brandId: String, transactionId: String?, isSuccess: Boolean, event: Event, errorCode: String?, errorMessage: String?, context: Context) {
        try {
            if (!isInitialized) {
                eventManagerPreferences.init(context)
            }
            if (transactionId.isNullOrEmpty()) {
                LPLog.d(TAG, "No transactionId found. Ignore event")
                return
            }
            val eventData = JSONObject()

            eventData.put(KEY_EVENT_TYPE, event.eventName)
            eventData.put(KEY_TIMESTAMP, System.currentTimeMillis())
            eventData.put(KEY_ACCOUNT_ID, brandId)
            eventData.put(KEY_TRANSACTION_ID, transactionId)
            eventData.put(KEY_CHANNEL, CHANNEL)
            eventData.put(KEY_VERSION, VERSION)
            eventData.put(KEY_REPORTER, REPORTER)
            eventData.put(KEY_VALUE, isSuccess)

            if (appId.isEmpty() && brandId.isNotEmpty()) {
                LPLog.d(TAG, "logEvent: Missing appId, trying to get it from shared preference.")
                appId = PreferenceManager.getInstance().getStringValue(PreferenceManager.APP_ID_PREFERENCE_KEY, brandId, "")
                eventData.put(KEY_OUTBOUND_NUMBER, appId)
            }

            // If we have 'conversation' event, add conversationId
            if (event == Event.CONVERSATION && !conversationId.isNullOrEmpty()) {
                eventData.put(KEY_CONVERSATION_ID, conversationId)
            }

            if (!errorCode.isNullOrEmpty()) {
                eventData.put(KEY_ERROR_CODE, errorCode)
            }

            if (!errorMessage.isNullOrEmpty()) {
                eventData.put(KEY_ERROR_MESSAGE, errorMessage)
            }

            LPLog.d(TAG, "Logging event '${event.eventName}' with transactionId: $transactionId")
            eventManagerPreferences.addEventToLogPreferences(eventData.toString(), brandId)

            // If we have conversation event, send it to manager immediately
            if (event == Event.CONVERSATION && !lpToken.isNullOrEmpty()) {
                sendLogsToEventManager(brandId, lpToken, arrayListOf(eventData), certificatePinningKeys)
                return
            }
        } catch (e: Exception) {
            LPLog.e(TAG, ErrorCode.ERR_0000015A, "Failed to log Event Manager event to log preferences")
        }
    }

    /**
     * Set conversationId when started new conversation. Applies only for 'conversation' event
     */
    fun setConversationId(conversationId: String?): EventManagerService {
        this.conversationId = conversationId
        return this
    }

    /**
     * Set event manager domain and immediately send cached events to Event Manager
     */
    fun setDomain(domain: String?) {
        if (this.domain.isNullOrEmpty() || this.domain != domain) {
            this.domain = domain
            if (!lpToken.isNullOrEmpty()) {
                sendCachedLogsToEventManager(brandId, lpToken, certificatePinningKeys)
            }
        }
    }

    /**
     * Listen to SDK's authentication state changes. Here, we only want to know If consumer is authenticated and has valid JWT
     */
    fun subscribeToAuthStateChanges(consumerManager: ConsumerManager) {
        consumerManager.subscribeToAuthStateChanges(object : AuthStateSubscription {
            override fun onAuthStateChanged(oldState: ConsumerManager.AuthState, newState: ConsumerManager.AuthState, oldConsumer: Consumer?, newConsumer: Consumer?) {

                if (newState == ConsumerManager.AuthState.AUTHENTICATED) {
                    if (newConsumer?.brandId != null && newConsumer.lpToken != null) {
                        lpToken = newConsumer.lpToken
                        certificatePinningKeys = newConsumer.lpAuthenticationParams?.certificatePinningKeys
                        sendCachedLogsToEventManager(newConsumer.brandId, lpToken, certificatePinningKeys)
                    }
                }
            }
        })
    }

    /**
     * If we have events stored in cache, fetch it and to event manager
     */
    fun sendCachedLogsToEventManager(brandId: String, token: String, certificatePinningKeys: MutableList<String>?) {
        val logs = eventManagerPreferences.fetchLogsFromPreferences(brandId)
        if (logs.isNullOrEmpty()) {
            LPLog.i(TAG, "sendLogsToEventManager: No logs found in cache.")
            return
        }
        val logList = ArrayList<JSONObject>()
        for (log in logs) {
            logList.add(JSONObject(log))
        }

        sendLogsToEventManager(brandId, token, logList, certificatePinningKeys)
    }

    /**
     * Sends events to event manager service.
     */
    private fun sendLogsToEventManager(brandId: String, token: String, logList: List<JSONObject>, certificatePinningKeys: MutableList<String>?) {
        try {
            if (lpToken.isNullOrEmpty()) {
                LPLog.e(TAG, ErrorCode.ERR_0000015B, "sendLogsToEventManager: Failed to send logs to Event Manager. Missing Token")
                return
            }
            if (domain.isNullOrEmpty()) {
                // Check If we have domain fetched from site settings
                domain = PreferenceManager.getInstance().getStringValue(PreferenceManager.SITE_SETTINGS_EVENT_MANAGER_DOMAIN_PREFERENCE_KEY, PreferenceManager.APP_LEVEL_PREFERENCES, null)
                if (domain.isNullOrEmpty()) {
                    LPLog.e(TAG, ErrorCode.ERR_0000015B, "sendLogsToEventManager: Failed to send logs to Event Manager. Missing domain")
                    return
                }
            }
            LPLog.d(TAG, "sendLogsToEventManager: Sending logs to event manager....")

            EventManagerRequest(domain, brandId, token, logList, certificatePinningKeys, object : ICallback<String, Exception> {
                override fun onSuccess(value: String?) {
                    LPLog.i(TAG, "sendLogsToEventManager: Successfully $value events by event manager. Clear events from cache")
                    eventManagerPreferences.clearLogsFromPreferences(brandId)
                }

                override fun onError(exception: Exception?) {
                    LPLog.e(TAG, ErrorCode.ERR_0000015B, "sendLogsToEventManager: Failed to send logs to event manager. {$exception}")
                }

            }).execute()
        } catch (error: Exception) {
            LPLog.e(TAG, ErrorCode.ERR_0000015B, "sendLogsToEventManager: Failed to send logs to event manager. {$error}")
        }
    }

    fun clearAllEvents() {
        eventManagerPreferences.clearAll()
    }
}