package com.liveperson.infra.network.http;


import android.support.annotation.NonNull;
import android.text.TextUtils;
import android.util.Pair;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.utils.TlsUtil;
import com.liveperson.infra.network.http.body.HttpRequestBody;
import com.liveperson.infra.network.http.request.HttpRequest;
import com.liveperson.infra.utils.ThreadPoolExecutor;
import com.liveperson.infra.utils.Utils;

import java.io.IOException;
import java.util.List;
import java.util.concurrent.TimeUnit;

import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.CertificatePinner;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;


/**
 * Created by ofira on 02/11/2015.
 */
public class HttpHandler {

    private static final String TAG = "HttpHandler";

	/**
	 * A list of OKHttp interceptors to be added to this handler
	 */
	private static List<Interceptor> mInterceptorList;

    /**
     * Execute http request with delay
     *
     * @param request - {@link HttpRequest}
     * @param delay   - long value to define delay
     */
    public static void executeDelayed(final HttpRequest request, long delay) {
        Runnable httpRunnable = buildRunnableByType(request);
        ThreadPoolExecutor.executeDelayed(httpRunnable, delay);
    }

    /**
     * Execute http request immediately
     *
     * @param request - {@link HttpRequest}
     */
    public static void execute(final HttpRequest request) {
        executeDelayed(request, 0);
    }

	/**
	 * Add a list of OKHttp interceptors to this HttpHandler
	 */
	public static void addInterceptors(List<Interceptor> interceptorList) {

		if ((interceptorList != null) && !interceptorList.isEmpty()) {
			LPLog.INSTANCE.d(TAG, "addInterceptors: adding interceptors to this HttpHandler");
			mInterceptorList = interceptorList;
		}
	}

    private static Runnable buildRunnableByType(final HttpRequest request) {
        return () -> {
            Request httpReq = HttpRequestBuilder.build(request);
            OkHttpClient.Builder clientBuilder = new OkHttpClient.Builder();
            TlsUtil.enableTls12ForKitKat(clientBuilder);
            clientBuilder.connectTimeout(request.getTimeout(), TimeUnit.MILLISECONDS);
            clientBuilder.writeTimeout(request.getTimeout(), TimeUnit.MILLISECONDS);
            clientBuilder.readTimeout(request.getTimeout(), TimeUnit.MILLISECONDS);
            clientBuilder.followRedirects(true);

            // Add all interceptors to clientBuilder
            if (mInterceptorList != null) {
                for (Interceptor interceptor : mInterceptorList) {
                    clientBuilder.addInterceptor(interceptor);
                }
            }

	        LPLog.INSTANCE.i(TAG, "URL: " + httpReq.url().host());

            //Certificate Pinning, checking if we have a valid list
            if (request.getCertificatePinningKeys() != null) {
                CertificatePinner.Builder builder = new CertificatePinner.Builder();
                for (String key : request.getCertificatePinningKeys()) {
	                LPLog.INSTANCE.d(TAG, "Pinning Key: " + LPLog.INSTANCE.mask(key));
                    if (Utils.isValidCertificateKey(key)) {
                        builder.add(httpReq.url().host(), key);
                    }
                }
                CertificatePinner certificatePinner = builder.build();
                clientBuilder.certificatePinner(certificatePinner);
            }

            OkHttpClient okClient = clientBuilder.build();

	        LPLog.INSTANCE.d(TAG, "Sending http request: " + httpReq.url() +
                     (request.getCertificatePinningKeys() != null ?  "with Pinning Keys " +  LPLog.INSTANCE.mask(TextUtils.join(",", request.getCertificatePinningKeys())) : " with no Pinning Keys"));

            okClient.newCall(httpReq).enqueue(new Callback() {
                @Override
                public void onFailure(@NonNull Call call, @NonNull IOException e) {
	                LPLog.INSTANCE.w(TAG, "onFailure ", e);
                    request.processErrorResponse(e);
                }

                @Override
                public void onResponse(@NonNull Call call, @NonNull Response response) {
	                LPLog.INSTANCE.i(TAG, "onResponse from URL: " + response.request().url());
                    request.processResponse(response);
                }
            });
        };
    }

    private static class HttpRequestBuilder {

        public static Request build(HttpRequest request) {
            Request.Builder httpBuilder = new Request.Builder();
            httpBuilder.url(request.getUrl());

            switch (request.getMethod()) {
                case GET:
                    httpBuilder.get();
                    break;
                case HEAD:
                    httpBuilder.head();
                    break;
                case POST:
                    httpBuilder.post(getRequestBody(request));
                    break;
                case PUT:
                    httpBuilder.put(getRequestBody(request));
                    break;
            }

            // Add headers
            if (!request.getHeaders().isEmpty()) {
                for (Pair<String, String> header : request.getHeaders()) {
                    LPLog.INSTANCE.d(TAG, "header.first " + header.first + " header.second " + header.second);
                    httpBuilder.addHeader(header.first, header.second);
                }
            }
            return httpBuilder.build();
        }

        @NonNull
        private static RequestBody getRequestBody(HttpRequest request) {
            MediaType type;
            RequestBody reqBody;
            HttpRequestBody requestBody = request.getRequestBody();
            if (requestBody != null) {
                type = MediaType.parse(requestBody.getContentType());
                if (requestBody.isString()) {
                    reqBody = RequestBody.create(type, (String) requestBody.get());
                } else {
                    reqBody = RequestBody.create(type, (byte[]) requestBody.get());
                }
            } else {
                reqBody = RequestBody.create(null, new byte[0]);
            }
            return reqBody;
        }
    }
}
