package com.liveperson.infra.network.http.requests;

import android.text.TextUtils;

import com.liveperson.infra.Command;
import com.liveperson.infra.ICallback;
import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.network.http.HttpHandler;
import com.liveperson.infra.network.http.request.HttpGetRequest;

import java.util.List;

import static com.liveperson.infra.errors.ErrorCode.ERR_0000004C;

/**
 * Created by nirni on 12/29/15.
 * <p/>
 * This is an HTTP request to the configuration from the site settings (AC-CDN)
 */
public class SiteSettingsRequest implements Command {

	private static final String TAG = "SiteSettingsRequest";
	private static final int REQUEST_TIMEOUT = 30000;

	// The URL to get the version
	private static final String SITE_SETTINGS_URL = "https://%s/api/account/%s/configuration/setting/accountproperties";

	private String mSiteSettingsDomain;
	private ICallback<String, Exception> mCallback;
	private String mBrandId;
	private List<String> mCertificates;

	public SiteSettingsRequest(String siteSettingsDomain, String brandId, List<String> certificates, ICallback<String, Exception> callback) {
		mSiteSettingsDomain = siteSettingsDomain;
		mCallback = callback;
		mBrandId = brandId;
		mCertificates = certificates;
	}

	@Override
	public void execute() {

		// Construct the URL to AC-CDN
		String siteSettingsUrl = String.format(SITE_SETTINGS_URL, mSiteSettingsDomain, mBrandId);

		HttpGetRequest httpGetRequest = new HttpGetRequest(siteSettingsUrl);
		httpGetRequest.setTimeout(REQUEST_TIMEOUT);
		httpGetRequest.setCertificatePinningKeys(mCertificates);
		httpGetRequest.setCallback(new ICallback<String, Exception>() {
			@Override
			public void onSuccess(String siteSettingsJsonString) {
				if (!TextUtils.isEmpty(siteSettingsJsonString)) {
					LPLog.INSTANCE.d(TAG, "siteSettingsRequest - onCompleted " + siteSettingsJsonString);
					mCallback.onSuccess(siteSettingsJsonString);
				}
				else {
					LPLog.INSTANCE.e(TAG, ERR_0000004C, "onSuccess: site settings string is empty");
					mCallback.onError(new Exception("site settings string is empty"));
				}
			}

			@Override
			public void onError(Exception e) {
				LPLog.INSTANCE.d(TAG, "Exception: ", e);
				mCallback.onError(e);
			}
		});
		HttpHandler.execute(httpGetRequest);
	}
}
