package com.liveperson.infra.log

import android.text.format.DateFormat
import com.liveperson.infra.utils.ExceptionUtils
import java.text.SimpleDateFormat
import java.util.*

/**
 * Data class to hold a single line of LivePerson log information.
 */
data class LogLine(val level: LogLevel, val tag: String, val message: String, val exception: Throwable? = null) {

	companion object {
		// Matches the format of Android's standard logcat timestamps.
		private const val DATE_FORMAT = "yyyy-MM-dd kk:mm:ss.SSS"
	}

	// Auto-generate timestamp
	val time = System.currentTimeMillis()

	// ToString overridden to print cleanly-formatted log lines.
	override fun toString(): String {
		val exceptionString = if (exception != null) { ExceptionUtils.getStringStackTrace(exception) } else { "" }

		// DateFormat only available on Android systems; SimpleDateFormat included for unit tests.
		val timeString = try {
			DateFormat.format(DATE_FORMAT, time)
		} catch (_: NoClassDefFoundError) {
			SimpleDateFormat(DATE_FORMAT, Locale.US).format(time)
		}

		// This line defines the format of log lines generated by this object.
		// exceptionString always starts with a newline if it isn't an empty-string.
		return "$timeString [${level.abbreviation}] [$tag] $message$exceptionString"
	}
}
