package com.liveperson.infra.utils

/*
 * This class adds three "global functions" to Kotlin within this codebase, supporting the 'unless'
 * block, and its companion block `otherwise`. The `unless` block is a fairly common language
 * feature wherein a boolean condition is evaluated inversely to the way an `if` block would
 * evaluate it; in other words, `unless (condition) === if (!condition)`. Kotlin, however, does not
 * include this function, so this class patches it in.
 */

/**
 * Basic `unless` Block (Inverse `if`)
 *
 * `unless` evaluates a boolean expression and, if that expression is false, executes
 * the block of code that follows.
 *
 * Typical use:
 * ```kotlin
 * unless (expressionThatShouldBeFalse) {
 *     someCodeToRunIfFalse()
 * }
 * ```
 *
 * @param bool A boolean expression to evaluate
 * @param block A block of code that will be executed only if `bool` is **false**.
 */
inline fun <T> unless(bool: Boolean, block: () -> T): T? = if (!bool) block() else null

/**
 * Inline form of `unless` Block (Inverse `if`)
 *
 * `unless` evaluates a boolean expression and, if that expression is false, executes
 * the block of code that follows.
 *
 * Typical use:
 * ```kotlin
 * { someCodeToRun() } unless (expression)
 * ```
 *
 * @param bool A boolean expression to evaluate
 */
infix fun <T> (() -> T).unless(bool: Boolean): T? = if (!bool) this() else null

/**
 * Basic `otherwise` block. (Pairs well with inline `unless`)
 *
 * `otherwise` takes the result of a block, and if that result is null, returns the
 * result of its own block instead.
 *
 * Typical use:
 * ```kotlin
 * return { "Stuff!" } unless (conditionIsTrue) otherwise { "Different stuff!" }
 * ```
 *
 * @param block A block of code that will be executed only if the input is null.
 */
inline infix fun <T> T?.otherwise(block: (() -> T)): T = this ?: block()
