package com.liveperson.infra.sdkstatemachine;

import com.liveperson.infra.log.LPLog;
import com.liveperson.infra.sdkstatemachine.events.InitEvent;
import com.liveperson.infra.sdkstatemachine.events.LogoutEvent;
import com.liveperson.infra.sdkstatemachine.events.PreLogoutCompletedEvent;
import com.liveperson.infra.sdkstatemachine.events.ShutDownCompletedEvent;
import com.liveperson.infra.sdkstatemachine.events.ShutDownEvent;
import com.liveperson.infra.sdkstatemachine.logout.PreLogoutCompletionListener;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDownCompletionListener;
import com.liveperson.infra.statemachine.BaseStateMachine;
import com.liveperson.infra.statemachine.InitProcess;
import com.liveperson.infra.statemachine.LogoutProcess;
import com.liveperson.infra.statemachine.ShutDownProcess;

import static com.liveperson.infra.errors.ErrorCode.*;

/**
 * Created by shiranr on 04/05/2016.
 */
public class InfraStateMachine extends BaseStateMachine {
    private static final String TAG = "InfraStateMachine";
    //STATES//
    private final Idle mIdleState = new Idle("IdleState", TAG + "_Idle");
    private final Initialized mInitializedState = new Initialized("Initialized", TAG);
    private final Initializing mInitializingState = new Initializing("Initializing", TAG);
    private final ShuttingDown mShuttingDown = new ShuttingDown("ShuttingDown", TAG);
    private final Logout mLoggingOutState = new Logout("Logout", TAG);

    public InfraStateMachine() {
        super(TAG);
        initActiveState(mIdleState);
    }

    public void initSDK(InitProcess initProcess) {
        if (initProcess == null) {
            LPLog.INSTANCE.e(TAG, ERR_00000009, "getInitProcess() is null!! error!");
            return;
        }
        postEvent(new InitEvent(initProcess));
    }

    public void shutDownSDK(ShutDownProcess shutDownProcess) {
        if (shutDownProcess == null) {
            LPLog.INSTANCE.e(TAG, ERR_0000000A, "getInitProcess() is null!! error!");
            return;
        }
        postEvent(new ShutDownEvent(shutDownProcess));
    }

    public void logoutSDK(LogoutProcess logoutProcess) {
        if (logoutProcess == null) {
            LPLog.INSTANCE.e(TAG, ERR_0000000B, "logoutProcess is null!! error!");
            return;
        }
        postEvent(new LogoutEvent(logoutProcess));
    }

    public boolean isSDKInitialized() {
        return (activeState()!= null && activeState().equals(mInitializedState));
    }

    /**
     * Idle- not initialized yet.
     */
    class Idle extends BaseInfraState{
        public Idle (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            changeStateAndPassEvent(mInitializingState, initEvent);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            changeStateAndPassEvent(mLoggingOutState, logoutEvent);
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            //ignore
            LPLog.INSTANCE.d(TAG, "shutDownEvent on init. ignore.");
        }
    }

    class Initializing extends InProgressState{

        public Initializing (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            InitProcess initProcess = initEvent.getInitProcess();

            try {
                initProcess.init();
                LPLog.INSTANCE.d(TAG, "Init finished Successfully! :) ");
                changeStateAndPassEvent(mInitializedState, initEvent);
            } catch (Exception e) {
                LPLog.INSTANCE.e(TAG, ERR_0000000C, "Exception while visiting InitEvent: " + initEvent, e);
                initProcess.getInitCallback().onInitFailed(e);
                changeState(mIdleState);
            }
        }
    }
    class Initialized extends BaseInfraState{
        public Initialized (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(InitEvent initEvent) {
            LPLog.INSTANCE.d(TAG, "Initialized! calling onInitSucceed callback");
            initEvent.getInitProcess().getInitCallback().onInitSucceed();
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            changeStateAndPassEvent(mShuttingDown, shutDownEvent);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            logoutEvent.setSkipInit(true);
            changeStateAndPassEvent(mLoggingOutState, logoutEvent);
        }
    }

    /**
     * Shutting down the SDK
     */
    class ShuttingDown extends InProgressState{

        private ShutDownProcess mShutDownProcess = null;
        private InitEvent mInitEventAfterShutDown = null;

        public ShuttingDown (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mShutDownProcess = null;
            mInitEventAfterShutDown = null;
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            //call super for logout event handling
            super.visit(shutDownEvent);
            //first time in this state
            if (mShutDownProcess == null){
                mShutDownProcess = shutDownEvent.getInitProcess();
                mShutDownProcess.shutDown(new ShutDownCompletionListener() {
                    @Override
                    public void shutDownCompleted() {
                        postEvent(new ShutDownCompletedEvent());
                    }

					@Override
					public void shutDownFailed() {

					}
				});
            }else{
                //already during shut down
                //shut down reset waiting init event
                LPLog.INSTANCE.d(TAG, "got shutDown event while processing Shutting Down... removing waiting init event if exists.");
                mInitEventAfterShutDown = null;
            }
        }

        @Override
        public void visit(ShutDownCompletedEvent shutDownCompletedEvent) {
	        LPLog.INSTANCE.d(TAG, "Shut down finished successfully! :) ");
            if (logoutEventWaiting != null){
                LPLog.INSTANCE.d(TAG, "Logout event waiting, logging out...");
                changeStateAndPassEvent(mLoggingOutState, logoutEventWaiting);
            }else if (mInitEventAfterShutDown != null){
                LPLog.INSTANCE.d(TAG, "Init event waiting, moving to initialized...");
                changeStateAndPassEvent(mInitializingState, mInitEventAfterShutDown);
            }else{
                changeState(mIdleState);
            }
        }

        @Override
        public void visit(InitEvent initEvent) {
            super.visit(initEvent);
            LPLog.INSTANCE.d(TAG, "got init event while processing Shutting Down...");
            mInitEventAfterShutDown = initEvent;
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            super.visit(logoutEvent);
            LPLog.INSTANCE.d(TAG, "Got logout event while processing Shutting Down... removing init waiting event");
            mInitEventAfterShutDown = null;
        }
    }

    /**
     * Logging out from SDK - removing all user data
     */
    class Logout extends BaseInfraState {
        private InitEvent mInitEventAfterLogout = null;
        private LogoutEvent mLogoutEvent;

        @Override
        public void actionOnEntry() {
            super.actionOnEntry();
            mInitEventAfterLogout = null;
            mLogoutEvent = null;
        }

        public Logout (String name, String logTag) {
            super(name, logTag);
        }

        @Override
        public void visit(LogoutEvent logoutEvent) {
            if (mLogoutEvent == null){
                mLogoutEvent = logoutEvent;
                mInitEventAfterLogout = logoutEvent.getInitEvent();
                logout();
            }else{
                //ignore
                LPLog.INSTANCE.d(TAG, "got logout event while processing logout.. ignoring event");
            }
        }

        @Override
        public void visit(InitEvent initEvent) {
            mInitEventAfterLogout = initEvent;
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            mInitEventAfterLogout = null;
        }

        @Override
        public void visit(PreLogoutCompletedEvent preLogoutCompletedEvent) {
            //shut down
            LPLog.INSTANCE.d(TAG, "shutDownForLogout...");
            LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            logoutProcess.shutDownForLogout(new ShutDownCompletionListener() {
                @Override
                public void shutDownCompleted() {
                    postEvent(new ShutDownCompletedEvent());
                }

				@Override
				public void shutDownFailed() {

				}
			});
        }

        @Override
        public void visit(ShutDownCompletedEvent shutDownCompletedEvent) {
            //logout
            LPLog.INSTANCE.d(TAG, "logout...");
            LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            logoutProcess.logout();
            onLogoutSucceed(logoutProcess);
        }

        private void logout() {
            LPLog.INSTANCE.d(TAG, "Stating logout process...");
            final LogoutProcess logoutProcess = mLogoutEvent.getLogoutProcess();
            try {
                if(!mLogoutEvent.skipInit()){
                    //init if needed
                    LPLog.INSTANCE.d(TAG, "initForLogout...");
                    logoutProcess.initForLogout();
                }

                // pre logout
                LPLog.INSTANCE.d(TAG, "preLogout...");
                logoutProcess.preLogout(new PreLogoutCompletionListener() {
                    @Override
                    public void preLogoutCompleted() {
                        postEvent(new PreLogoutCompletedEvent());
                    }

                    @Override
                    public void preLogoutFailed(Exception e) {
                        //notify listener logout failed.
                        LPLog.INSTANCE.w(TAG, "error while preLogoutFailed: ", e);
                        onLogoutFailed(e, logoutProcess);

                    }
                });

            } catch (Exception e) {
                LPLog.INSTANCE.w(TAG, "error while logout: ", e);
                onLogoutFailed(e, logoutProcess);
            }
        }

        private void onLogoutSucceed(LogoutProcess logoutProcess) {
            logoutProcess.getLogoutCallback().onLogoutSucceed();
            exitState();
        }

        private void onLogoutFailed(Exception e, LogoutProcess logoutProcess) {
            logoutProcess.getLogoutCallback().onLogoutFailed(e);
            exitState();
        }

        private void exitState() {
            if (mInitEventAfterLogout != null) {
                changeStateAndPassEvent(mIdleState, mInitEventAfterLogout);
            }else{
                changeState(mIdleState);
            }
        }

    }


    abstract class InProgressState extends BaseInfraState{
        LogoutEvent logoutEventWaiting = null;

        public InProgressState (String name, String logTag) {
            super( name, logTag);
        }

        @Override
        public void actionOnEntry() {
            logoutEventWaiting = null;
        }

        @Override
        public void visit(InitEvent initEvent) {
            if (logoutEventWaiting != null){
                logoutEventWaiting.setInitAfterLogout(initEvent);
                LPLog.INSTANCE.d(TAG, "Logout event waiting, init after logout is NOT null ");
            }
        }

        @Override
        public void visit(ShutDownEvent shutDownEvent) {
            if (logoutEventWaiting != null){
                logoutEventWaiting.setInitAfterLogout(null);
                LPLog.INSTANCE.d(TAG, "Logout event waiting, init after logout is null ");
            }
        }
        @Override
        public void visit(LogoutEvent logoutEvent) {
            logoutEventWaiting = logoutEvent;
            logoutEventWaiting.setInitAfterLogout(null);
            LPLog.INSTANCE.d(TAG, "Logout event waiting, init after logout is null ");
        }
    }
}
