package com.liveperson.infra.network.socket;


import android.text.TextUtils;

import com.liveperson.infra.Infra;
import com.liveperson.infra.model.SocketConnectionParams;
import com.liveperson.infra.sdkstatemachine.shutdown.ShutDown;
import com.liveperson.infra.log.LPMobileLog;
import com.liveperson.infra.network.socket.state.SocketStateListener;

import java.util.HashMap;
import java.util.Map;

/**
 * Created by Ilya Gazman on 11/3/2015.
 */
public class SocketManager implements ShutDown {

    private static final String TAG = SocketManager.class.getSimpleName();
    private final Object mSyncObject = new Object();
    private final ResponseMap mResponseMap = new ResponseMap();
    private Map<String, SocketHandler> mSocketHandlersMap;


    private static volatile SocketManager Instance = null;


    private SocketManager() {
        mSocketHandlersMap = new HashMap<>();
    }


    /**
     * Get instance of the infra Preference Manager
     * Note: this class uses the application context initialized in the {@link Infra} class
     *
     * @return
     */
    public static SocketManager getInstance() {
        if (Instance == null) {
            synchronized (SocketManager.class) {
                if (Instance == null) {
                    Instance = new SocketManager();
                }
            }
        }
        return Instance;
    }

    /**
     * Will get a new socket and start connection process
     * if socket already exist, it will be used to resume connection
     *
     * @param connectionParams
     */
    public void connect(SocketConnectionParams connectionParams) {
        if (TextUtils.isEmpty(connectionParams.getUrl())) {
            LPMobileLog.e(TAG, "Can't connect to empty url");
            return;
        }
        LPMobileLog.i(TAG, "connecting to socket");
        SocketHandler socketHandler = obtainSocket(connectionParams.getUrl());
        socketHandler.connect(connectionParams);
    }

    /**
     * Will get a new socket and start connection process
     * if socket already exist, it will be used to resume connection
     *
     * Deprecated since 11/12/2017 //TODO: Perry: "Create and use a smart annotation that will help us know exactly when it becomes one year old. For example: @LPDeprecated(since = 1512950400) // While the 'since' is in seconds (int, not long)"
     *
     * @param url connection url
     * @Deprecated      //TODO: Perry: "Is this should 'annotation' be on THIS line or it was made by mistake?"
     */
    @Deprecated
    public void connect(final String url, final int timeout) {
        if (TextUtils.isEmpty(url)) {
            LPMobileLog.e(TAG, "Can't connect to empty url");
            return;
        }
        LPMobileLog.i(TAG, "connecting to socket");
        SocketConnectionParams params = new SocketConnectionParams(url, timeout);
        connect(params);
    }

    /**
     * Create a new socket and put it in a local map
     * In case we already have existing socket, we return it
     *
     * @param url
     * @return
     */
    private SocketHandler obtainSocket(final String url) {
        if (TextUtils.isEmpty(url)) {
            LPMobileLog.e(TAG, "Can't obtain socket! url is empty");
            return null;
        }
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                socketHandler = new SocketHandler(mResponseMap);
                mSocketHandlersMap.put(url, socketHandler);
            }
            return socketHandler;
        }
    }

    /**
     * Send request over the socket
     *
     * @param baseSocketRequest
     */
    public void send(BaseSocketRequest baseSocketRequest) {
        SocketHandler socketHandler = getSocketHandler(baseSocketRequest);
        if (socketHandler != null) {
            LPMobileLog.i(TAG, "Sending request " + baseSocketRequest.getRequestName());
            String data = baseSocketRequest.getData();
            //data = data.replace("\\", "");
            socketHandler.send(data);
        }
    }

    /**
     * Send request over the socket with initial delay
     *
     * @param baseSocketRequest
     * @param delay
     * @return
     */
    public Runnable send(BaseSocketRequest baseSocketRequest, long delay) {
        SocketHandler socketHandler = getSocketHandler(baseSocketRequest);
        if (socketHandler != null) {
            return socketHandler.sendDelayed(baseSocketRequest.getData(), delay);
        } else {
            LPMobileLog.d(TAG, "Can't schedule send delayed SocketRequest");
        }
        return null;
    }

    /**
     * Get the
     *
     * @param baseSocketRequest
     * @return
     */
    private SocketHandler getSocketHandler(BaseSocketRequest baseSocketRequest) {
        if (baseSocketRequest == null) {
            LPMobileLog.e(TAG, "can't get SocketHandler with null request");
            return null;
        }
        LPMobileLog.d(TAG, "getSocketHandler req id " + baseSocketRequest.getRequestId());
        BaseResponseHandler responseHandler = baseSocketRequest.getResponseHandler();
        if (responseHandler != null) {
            LPMobileLog.d(TAG, "getResponseHandler for request " + baseSocketRequest.getRequestId() + " is not null");
            responseHandler.init(baseSocketRequest);
            mResponseMap.putRequestIdHandler(baseSocketRequest.getRequestId(), responseHandler);
        }
        return obtainSocket(baseSocketRequest.getSocketUrl());
    }

    /**
     * @param baseSocketRequest
     * @param request
     */
    public void cancelDelayedRequest(BaseSocketRequest baseSocketRequest, Runnable request) {
        if (baseSocketRequest == null) {
            LPMobileLog.e(TAG, "Can't run null request!");
            return;
        }
        SocketHandler socketHandler = obtainSocket(baseSocketRequest.getSocketUrl());
//        if (socketHandler != null) {
//            socketHandler.cancelRunnable(request);
//        }
    }

    /**
     * Silently kill the socket, it will also remove any callbacks
     *
     * @param url connection url
     */
/*    public void disconnect(String url) {
        LPMobileLog.i(TAG, "disconnect " + url);
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            LPMobileLog.i(TAG, "remove from the local map " + url);
            mSocketHandlersMap.remove(url);
            if (socketHandler == null) {
                return;
            }
            socketHandler.dispose();
        }
    }*/


    /**
     * Disconnecting the socket, anc clear any callbacks or pending messages
     *
     * @param url connection url
     */
    public void killSocket(String url) {
        LPMobileLog.i(TAG, "kill socket");
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return;
            }
            socketHandler.disconnect();
            socketHandler.dispose();
        }
    }


    /**
     * Disconnecting the socket
     *
     * @param url connection url
     */
    public void disconnect(String url) {
        LPMobileLog.d(TAG, "disconnect " + url);
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return;
            }
            socketHandler.disconnect();
        }
    }

    @Override
    public void shutDown() {
        LPMobileLog.i(TAG, "Shutting down all");
        for (String connectionUrl : mSocketHandlersMap.keySet()) {
            killSocket(connectionUrl);
        }
        mSocketHandlersMap.clear();
        mResponseMap.shutDown();
        Instance = null;
    }

    /**
     * Allow you to register to socket callbacks, like error and data.
     * You can call this method right after connect
     *
     * @param url connection url
     */
    public void registerToSocketState(String url, SocketStateListener callback) {
        SocketHandler socketHandler = obtainSocket(url);
        if (socketHandler != null) {
            socketHandler.getSocketStateManager().register(callback);
        }
    }

    /**
     * unregister from socket state changes callbacks
     *
     * @param url connection url
     */
    public void unregisterFromSocketState(String url, SocketStateListener callback) {
        SocketHandler socketHandler;
        synchronized (mSyncObject) {
            socketHandler = mSocketHandlersMap.get(url);
        }
        if (socketHandler != null) {
            socketHandler.getSocketStateManager().unregister(callback);
        }
    }

    /**
     * Get socket state
     *
     * @param url connection url
     * @return the current state of the socket, or INIT in case there is no active socket
     */
    public SocketState getSocketState(String url) {
        synchronized (mSyncObject) {
            SocketHandler socketHandler = mSocketHandlersMap.get(url);
            if (socketHandler == null) {
                return SocketState.INIT;
            }
            SocketStateManager socketStateManager = socketHandler.getSocketStateManager();
            if (socketStateManager == null) {
                return SocketState.INIT;
            }
            return socketStateManager.getState();
        }
    }

    public void putGeneralHandlerMap(GeneralResponseHandler generalResponseHandler) {
        mResponseMap.putGeneralHandler(generalResponseHandler);
    }
}
