package com.liveperson.infra.utils;

import android.Manifest;
import android.content.Context;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.media.AudioManager;
import android.media.MediaPlayer;
import android.media.MediaRecorder;
import android.media.SoundPool;
import android.os.Build;
import androidx.annotation.Nullable;
import androidx.core.content.ContextCompat;
import android.text.TextUtils;
import android.view.KeyEvent;

import com.liveperson.infra.ICallback;
import com.liveperson.infra.Infra;
import com.liveperson.infra.controller.PlayingAudioManager;
import com.liveperson.infra.log.LPMobileLog;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;

import static android.view.KeyEvent.KEYCODE_MEDIA_PAUSE;
import static android.view.KeyEvent.KEYCODE_MEDIA_PLAY;

/**
 * Created by perrchick on 21/11/2017. Modified by nirni
 * An adapter that adapts: https://developer.android.com/reference/android/media/MediaRecorder.html
 */
public class LPAudioUtils {

	public static final String TAG = "LPAudioUtils";

	private final File mFilesFolder;

	public enum LPRecorderStatus {
		PermissionDenied, Failed, Started, Finished
	}

	public static String VOICE_FOLDER = "voice/";
	private static final String AUDIO_RECORDING_STARTED_BROADCAST = "AUDIO_RECORDING_STARTED_BROADCAST";
	private static final String AUDIO_RECORDING_STOPPED_BROADCAST = "AUDIO_RECORDING_STOPPED_BROADCAST";

	public interface PlaybackCallback {
		void onPlaybackStarted(String audioFileUrl, int durationMilliseconds);
		void onPlaybackCompleted(boolean successfully, String audioFileUrl);
	}

	private final HashMap<String, PlaybackCallback> mCallbacks;

	private SoundPool mSoundPool;
	@Nullable
	private LPMediaPlayer mMediaPlayer;
	@Nullable
	private MediaRecorder mMediaRecorder;

	private File mTempRecordedAudioFile;
	private RecordingResultCallback recordingCallback;

	/**
	 * An audio playing manager that holds all currently playing audio
	 */
	private PlayingAudioManager mPlayingAudioManager;

	/**
	 * The system AudioManager service
	 */
	private AudioManager mAudioManager;

	/**
	 * Indicate whether another audio is active when we started recording/playing
	 */
	private boolean mOtherAudioActive;

	/**
	 * BroadcastReceiver with action to take when headset is unplugged
	 */
	private DetectHeadsetUnpluggedBroadcastReceiver mDetectHeadsetUnpluggedBroadcastReceiver = new DetectHeadsetUnpluggedBroadcastReceiver() {
		@Override
		protected void onUnpluggedHeadset() {
			mPlayingAudioManager.stopAllCurrentlyPlaying();
			stopPlayback();
		}
	};


	public LPAudioUtils() {
		// Playback won't be able to play more then 5 simultaneously
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
			mSoundPool = new SoundPool.Builder().setMaxStreams(5).build();
		} else {
			mSoundPool = new SoundPool(5, AudioManager.STREAM_VOICE_CALL, 0); // Or should we use 'AudioManager.STREAM_MUSIC' ??
		}
		mCallbacks = new HashMap<>();

		mFilesFolder = Infra.instance.getApplicationContext().getFilesDir();

		mPlayingAudioManager = new PlayingAudioManager();

		mAudioManager = (AudioManager) Infra.instance.getApplicationContext().getSystemService(Context.AUDIO_SERVICE);

		mOtherAudioActive = false;

	}
	//region

	@Nullable
	private MediaRecorder initMediaRecorder() {
		if (mMediaRecorder != null) {
			mMediaRecorder.release();
		}

		MediaRecorder mediaRecorder = new MediaRecorder();
		mediaRecorder.setAudioChannels(1);
		mediaRecorder.setAudioSource(MediaRecorder.AudioSource.MIC);
		mediaRecorder.setOutputFormat(MediaRecorder.OutputFormat.MPEG_4);
		mediaRecorder.setAudioEncoder(MediaRecorder.AudioEncoder.AAC);
		mediaRecorder.setAudioSamplingRate(16000);

		return mediaRecorder;
	}

	@Nullable
	private LPMediaPlayer initMediaPlayer(String callbackId, String localFile, MediaPlayer.OnCompletionListener onCompletionListener) {
		if (mMediaPlayer != null) {
			mMediaPlayer.release();
		}

		LPMediaPlayer mediaPlayer = new LPMediaPlayer(callbackId, localFile);
		mediaPlayer.setOnCompletionListener(onCompletionListener);
		return mediaPlayer;
	}

	/**
	 * Start recording and use the given filename
	 *
	 * @param filename                the filename to save the recording to
	 * @param maxDurationMs           the maximum time in milliseconds for this recording. The recording automatically stopped when reached the maximum and the given recordingResultCallback is called
	 * @param recordingResultCallback a listener to be called when maximum recording time is reached
	 * @return
	 */
	public LPRecorderStatus startRecording(String filename, int maxDurationMs, final RecordingResultCallback recordingResultCallback) {
		LPMobileLog.d(TAG, "startRecording: start recording with max duration (ms) : " + maxDurationMs);

		Context appContext = Infra.instance.getApplicationContext();
		LPRecorderStatus recorderStatus = LPRecorderStatus.PermissionDenied;
		boolean isAudioRecordingPermissionGranted = ContextCompat.checkSelfPermission(appContext, Manifest.permission.RECORD_AUDIO) == PackageManager.PERMISSION_GRANTED;
		if (!isAudioRecordingPermissionGranted) {
			return recorderStatus;
		}

		recorderStatus = LPRecorderStatus.Failed;
		mTempRecordedAudioFile = new File(getVoiceFolder(), filename);
		recordingCallback = recordingResultCallback;
		mMediaRecorder = initMediaRecorder();
		if (mMediaRecorder == null) {
			return recorderStatus;
		}

		boolean didStartRecording = false;

		mMediaRecorder.setOutputFile(mTempRecordedAudioFile.getPath());

		// Set maximum recording duration
		mMediaRecorder.setMaxDuration(maxDurationMs);
		// Call the listener if max recording duration reached
		mMediaRecorder.setOnInfoListener(new MediaRecorder.OnInfoListener() {
			@Override
			public void onInfo(MediaRecorder mediaRecorder, int what, int extra) {
				if (what == MediaRecorder.MEDIA_RECORDER_INFO_MAX_DURATION_REACHED) {
					LPMobileLog.d(TAG, "onInfo: maximum recoding time reached. Stop the recording and call the callback");
					stopRecording(new ICallback<String, Exception>() {
						@Override
						public void onSuccess(String audioFilePath) {
							if (recordingCallback != null) {
								recordingCallback.onMaxRecordingDurationReached(audioFilePath);
								recordingCallback = null;
							}
						}

						@Override
						public void onError(Exception exception) {
							if (recordingCallback != null) {
								recordingCallback.onMaxRecordingDurationReached(null);
								recordingCallback = null;
							}
						}
					});
				}
			}
		});

		try {

			pauseExternalAudio();

            mMediaRecorder.prepare();
            mMediaRecorder.start();
            didStartRecording = true;
        } catch (Throwable exception) {
            LPMobileLog.e(TAG, "failed to stop audio record", exception);
        }

        if (didStartRecording) {
            LocalBroadcast.sendBroadcast(AUDIO_RECORDING_STARTED_BROADCAST);
            recorderStatus = LPRecorderStatus.Started;
        }

        return recorderStatus;
    }

	/**
	 * Stop recording and save the file. Filename is returned on the given callback
	 * @param resultCallback
	 */
	public void stopRecording(ICallback<String, Exception> resultCallback) {

		LPMobileLog.d(TAG, "stopRecording: stop recording");

        if (!isRecording() || mTempRecordedAudioFile == null || mMediaRecorder == null) {
			if (resultCallback != null) {
				resultCallback.onError(new Exception("missing recorded file"));
			}
            return;
        }

        String tempAudioFilePath = mTempRecordedAudioFile.getPath();

		LPMobileLog.d(TAG, "stopRecording: recording file path: " + tempAudioFilePath);

        try {
            mMediaRecorder.stop();
            mMediaRecorder.release();

			continueExternalAudio();

			if (resultCallback != null) {
				resultCallback.onSuccess(tempAudioFilePath);
			}
			if (recordingCallback != null) {
				recordingCallback.onRecordingInterrupted(tempAudioFilePath);
			}
            LocalBroadcast.sendBroadcast(AUDIO_RECORDING_STOPPED_BROADCAST);

        } catch (IllegalStateException e) {
            LPMobileLog.e(TAG, "failed to stop audio record", e);
        } finally {
            mMediaRecorder = null;
            mTempRecordedAudioFile = null;
            recordingCallback = null;
        }
    }

	/**
	 * Get the duration of the given file in milliseconds. The value returns using the given callback
	 * @param filePath
	 * @param resultCallback
	 */
	public static void getDuration(String filePath, final ICallback<Integer, Exception> resultCallback) {
        if (resultCallback == null) return;

        if (TextUtils.isEmpty(filePath)) {
            resultCallback.onError(new Exception("file path is empty"));
            return;
        }
        MediaPlayer mediaPlayer = new MediaPlayer();
        try {
            mediaPlayer.setDataSource(filePath);
            mediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
                @Override
                public void onPrepared(MediaPlayer mp) {
                    resultCallback.onSuccess(mp.getDuration());
                }
            });
            mediaPlayer.prepareAsync();
        } catch (IOException e) {
			LPMobileLog.w(TAG, "getDuration: error getting duration of file " + filePath, e);
            resultCallback.onError(e);
        }
    }

	/**
	 * Bind the currently playing audio file to the given callback
	 * @param callbackId
	 * @param playbackCallback
	 */
	public void bindPlayingAudio(final String callbackId, PlaybackCallback playbackCallback) {
		mCallbacks.put(callbackId, playbackCallback);
	}

    public void playAudio(final String audioFileUrl, final String callbackId, PlaybackCallback playbackCallback) {
		if (playbackCallback == null) {
			return;
		}

		if (mMediaPlayer != null) {
			if (mMediaPlayer.isPlaying(audioFileUrl)) {
				return;
			}

			if (mMediaPlayer.getLocalFileUrl().equals(audioFileUrl)) {
				pauseExternalAudio();

				mMediaPlayer.start();
				playbackCallback.onPlaybackStarted(audioFileUrl, mMediaPlayer.getDuration());
				return;
			}

			if (!mMediaPlayer.getLocalFileUrl().equals(audioFileUrl) && mCallbacks.containsKey(mMediaPlayer.callbackId)) {
				PlaybackCallback callback = mCallbacks.remove(mMediaPlayer.callbackId);
				if (callback != null) {
					callback.onPlaybackCompleted(false, mMediaPlayer.getLocalFileUrl());
				}
			}
			// restart
			mMediaPlayer.release();
			mMediaPlayer = null;
		}

		if (isRecording()) {
			stopRecording(null);
		}

		if (TextUtils.isEmpty(audioFileUrl)) {
			playbackCallback.onPlaybackCompleted(false, audioFileUrl);
			return;
		}

		File tempPlayedAudioFile = new File(audioFileUrl);

		if (tempPlayedAudioFile.exists()) {
			mCallbacks.put(callbackId, playbackCallback);
			mMediaPlayer = initMediaPlayer(callbackId, audioFileUrl, new MediaPlayer.OnCompletionListener() {
				@Override
				public void onCompletion(MediaPlayer mp) {
					if (!(mp instanceof LPMediaPlayer)) return;
					LPMediaPlayer mediaPlayer = (LPMediaPlayer) mp;
					PlaybackCallback callback = mCallbacks.remove(mediaPlayer.getCallbackId());
					String _audioFileUrl = audioFileUrl;

					if (mediaPlayer == mMediaPlayer) {
						cleanupPlayback();
						_audioFileUrl = mediaPlayer.getLocalFileUrl();
						LPMobileLog.d(TAG, "Playback completed: " + _audioFileUrl);
					}

					if (callback != null) {
						callback.onPlaybackCompleted(true, _audioFileUrl);
					}
				}
			});

			if (mMediaPlayer == null) {
				tempPlayedAudioFile = null;
				return;
			}

			try {
				mMediaPlayer.setDataSource(audioFileUrl);
				mMediaPlayer.setOnPreparedListener(new MediaPlayer.OnPreparedListener() {
					@Override
					public void onPrepared(MediaPlayer mp) {
						if (mMediaPlayer == mp) {
							pauseExternalAudio();

							mp.start();

							// Register a receiver to listen to headset unplugged
							LPMobileLog.d(TAG, "onPrepared: Registering to detect unplugged headset");
							IntentFilter intentFilter = new IntentFilter(AudioManager.ACTION_AUDIO_BECOMING_NOISY);
							Infra.instance.getApplicationContext().registerReceiver(mDetectHeadsetUnpluggedBroadcastReceiver, intentFilter);

							PlaybackCallback callback = mCallbacks.get(mMediaPlayer.getCallbackId());
							if (callback != null) {
								callback.onPlaybackStarted(audioFileUrl, mp.getDuration());
							}
						}
					}
				});
				mMediaPlayer.prepareAsync();
			} catch (IOException e) {
				LPMobileLog.e(TAG, "Exception while opening data source with media player.", e);
			}
		}
	}

	/**
	 * Return the voice folder, create it if not exist
	 * @return the voice folder path or null if cannot does not exist and cannot be created
	 */
	@Nullable
	public String getVoiceFolder() {
		File outgoingAudioFolder = new File(mFilesFolder + "/" + VOICE_FOLDER);
		if (!outgoingAudioFolder.exists()) {
			if (!outgoingAudioFolder.mkdirs()) {
				LPMobileLog.e(TAG, "getVoiceFolder: Image folder could not be created");
				return null;
			}
		}

		return outgoingAudioFolder.getPath();
	}


	private void cleanupPlayback() {
        LPMediaPlayer mp = mMediaPlayer;
        mMediaPlayer = null;
        if (mp != null) {
            mp.stop();
            mp.release();

            continueExternalAudio();

			// Unregister a receiver
			try{
				Infra.instance.getApplicationContext().unregisterReceiver(mDetectHeadsetUnpluggedBroadcastReceiver);
			} catch (IllegalArgumentException e) { // In case the receiver is not registered
				LPMobileLog.w(TAG, "cleanupPlayback: receiver is not registered", e);
			}

            mCallbacks.remove(mp.getCallbackId());
        }
    }

    public boolean isRecording() {
        return (mTempRecordedAudioFile != null) && (mMediaRecorder != null);
    }

	/**
	 * Return the current playing location
	 * @return
	 */
	public int getCurrentPlayingLocation(){
		int location = 0;
		if (mMediaPlayer != null && mMediaPlayer.isPlaying()) {
			location = mMediaPlayer.getCurrentPosition();
		}
		return location;
	}

	/**
	 * Return the current playing duration. If currently not playing return -1
	 * @return
	 */
	public int getCurrentPlayingDuration() {
		int duration = -1;
		if (mMediaPlayer != null && mMediaPlayer.isPlaying()) {
			duration = mMediaPlayer.getDuration();
		}
		return duration;
	}

    private void playSound(String audioFilePath) {
        final Integer soundId = mSoundPool.load(audioFilePath, 1);
        final int currentVolume = getCurrentVolume();
        mSoundPool.setOnLoadCompleteListener(new SoundPool.OnLoadCompleteListener() {
            @Override
            public void onLoadComplete(SoundPool soundPool, int sampleId, int status) {
                if (sampleId == soundId) {
                    soundPool.play(soundId, currentVolume, currentVolume, 1, 0, 1f);
                }
            }
        });
    }

    private int getCurrentVolume() {
        Context appContext = Infra.instance.getApplicationContext();
        AudioManager audioManager = (AudioManager) appContext.getSystemService(Context.AUDIO_SERVICE);
        int currentVolume = audioManager.getStreamVolume(AudioManager.STREAM_NOTIFICATION);
        return currentVolume;
    }

    public void playSound(int resourceId) {
        final Integer soundId = mSoundPool.load(Infra.instance.getApplicationContext(), resourceId, 1);
        final int currentVolume = getCurrentVolume();
        mSoundPool.setOnLoadCompleteListener(new SoundPool.OnLoadCompleteListener() {
            @Override
            public void onLoadComplete(SoundPool soundPool, int sampleId, int status) {
                if (sampleId == soundId) {
                    soundPool.play(soundId, currentVolume, currentVolume, 1, 0, 1f);
                }
            }
        });
    }

	public void stopPlayback() {
		if (mMediaPlayer != null) {

			if (mMediaPlayer.isCurrentlyPlaying() && mMediaPlayer.getLocalFileUrl() != null &&
					mMediaPlayer.callbackId != null && mCallbacks.containsKey(mMediaPlayer.callbackId)) {

				PlaybackCallback callback = mCallbacks.remove(mMediaPlayer.callbackId);
				if (callback != null) {
					callback.onPlaybackCompleted(false, mMediaPlayer.getLocalFileUrl());
				}
			}

			cleanupPlayback();
		}
	}

	public PlayingAudioManager getPlayingAudioManager() {
		return mPlayingAudioManager;
	}

	/**
	 * Save the given byte array to disk to the given file
	 *
	 * @param fileByteArray
	 * @return path to saved file
	 */
	public String saveByteArrayToDisk(byte[] fileByteArray) {

		// Generate filename
		File filePath = new File(getVoiceFolder(), generateVoiceFileName());


		LPMobileLog.d(TAG, "saveByteArrayToDisk: filePath: " + filePath.getAbsolutePath());

		FileOutputStream fos = null;
		try {
			fos = new FileOutputStream(filePath);
			fos.write(fileByteArray);

		} catch (FileNotFoundException e) {
			LPMobileLog.e(TAG, "saveByteArrayToDisk: File not found", e);
			return null;

		} catch (IOException e) {
			LPMobileLog.e(TAG, "saveByteArrayToDisk: IOException", e);

		} finally {
			try {
				if (fos != null) {
					fos.close();
				}
			} catch (IOException e) {
				LPMobileLog.e(TAG, "saveByteArrayToDisk: error closing file", e);
				return null;
			}
		}

		LPMobileLog.d(TAG, "saveByteArrayToDisk: file absolute path: " + filePath.getAbsolutePath());
		return filePath.getAbsolutePath();
	}

	public static String generateVoiceFileName(){

		return UniqueID.createUniqueMessageEventId() + ".m4a";
	}

	/**
	 * Pause any other music player currently playing
	 */
	private void pauseExternalAudio() {
		if (mAudioManager.isMusicActive()) {
			mOtherAudioActive = true;

			LPMobileLog.d(TAG, "pauseExternalAudio: other audio is playing. Pausing it...");

			KeyEvent event = new KeyEvent(KeyEvent.ACTION_DOWN, KEYCODE_MEDIA_PAUSE);
			mAudioManager.dispatchMediaKeyEvent(event);
			event = new KeyEvent(KeyEvent.ACTION_UP, KEYCODE_MEDIA_PAUSE);
			mAudioManager.dispatchMediaKeyEvent(event);
		}
		else {
			mOtherAudioActive = false;
		}
	}

	/**
	 * If we paused any other audio playing, continue playing it
	 */
	private void continueExternalAudio() {

		if (mOtherAudioActive) {

			LPMobileLog.d(TAG, "continueExternalAudio: Replaying other audi");

			KeyEvent event = new KeyEvent(KeyEvent.ACTION_DOWN, KEYCODE_MEDIA_PLAY);
			mAudioManager.dispatchMediaKeyEvent(event);
			event = new KeyEvent(KeyEvent.ACTION_UP, KEYCODE_MEDIA_PLAY);
			mAudioManager.dispatchMediaKeyEvent(event);
		}
	}

	/**
	 * Returns true in case the file is currently being played / paused
	 *
	 * @param audioFilePath
	 * @return True in case the audio file is currently being played / paused
	 */
	public boolean isPlaying(String audioFilePath) {
		return mMediaPlayer != null && mMediaPlayer.isPlaying(audioFilePath);
	}

	public void pause() {
		if (mMediaPlayer != null) {
			mMediaPlayer.pause();
		}
	}

	private class LPMediaPlayer extends MediaPlayer {
		private final String localFileUrl;
		private final String callbackId;

		public LPMediaPlayer(String callbackId, String localFileUrl) {
			super();
			this.callbackId = callbackId;
			this.localFileUrl = localFileUrl;
		}

		public String getCallbackId() {
			return callbackId;
		}

		public String getLocalFileUrl() {
			return localFileUrl;
		}

		public boolean isPlaying(String audioFileUrl) {
			return isPlaying() && !TextUtils.isEmpty(audioFileUrl) && audioFileUrl.equals(localFileUrl);
		}

		public boolean isCurrentlyPlaying() {
			return isPlaying();
		}
	}

	/**
	 * Interface to inform that the recording has ended before the caller requested that it end.
	 * These callback methods return the audio file path, or null in case recording cannot be saved.
	 */
	public interface RecordingResultCallback {
		/**
		 * Called when the maximum duration for a recorded clip is reached.
		 * @param audioFilePath The path of the file where the recorded audio was saved.
		 */
		void onMaxRecordingDurationReached(@Nullable String audioFilePath);

		/**
		 * Called if something interrupts the recording midway, such as playing an audio clip.
		 * @param audioFilePath The path of the file where the recorded audio was saved.
		 */
		void onRecordingInterrupted(@Nullable String audioFilePath);
	}
}
