package com.liveperson.infra.log;

import androidx.annotation.NonNull;

import com.liveperson.infra.BuildConfig;
import com.liveperson.infra.Infra;
import com.liveperson.infra.log.logreporter.loggos.Loggos;
import com.liveperson.infra.log.logreporter.loggos.LoggosMessage;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.io.Writer;

/**
 * Log helper.
 * Should be in use only by LP SDK, not by the host app or others.
 *
 * @author ofira
 */
public class LPMobileLog {

	private static final String TAG = "LivePerson";

	private static boolean DEBUG = true; //TODO: use BuildConfig.DEBUG

	private static boolean isLoggosEnabled = true;
	private static final String LOGGOS_CONTEXT = "AndroidSdk";
	private static final String INFO = "INFO";
	private static final String WARN = "WARN";
	private static final String ERROR = "ERROR";

	private static Logger logger = new AndroidLogger();

	public static void setLogger(Logger logger) {
		LPMobileLog.logger = logger;
	}

	/** Set If we want to write logs to Loggos.
	 * This is false If we are running in test environment.
	 *
	 *  @param enabled <b>true</b> if you want to send logs to Loggos; <b>false</b> otherwise.
	 **/
	public static void setLoggosEnabled(boolean enabled) {
		isLoggosEnabled = enabled;
	}

	public static void setDebugMode(boolean debugMode) {
		LPMobileLog.DEBUG = debugMode;
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	public static void v(@NonNull String tag, @NonNull String message) {
		if (DEBUG) {
			logger.v(TAG, "[[" + tag + "]]  " + message);
		}
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	public static void v(@NonNull String tag, @NonNull String flowTag, @NonNull String message) {
		if (DEBUG) {
			logger.v(TAG, "[[" + tag + "::" + flowTag + "]]  " + message);
		}
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void v(@NonNull String tag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.v(TAG, "[[" + tag + "]]  " + message, exception);
		}
	}

	/**
	 * Logs a message using the lowest developer-facing log-level (VERBOSE).
	 * Good for mindless log spam.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void v(@NonNull String tag, @NonNull String flowTag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.v(TAG, "[[" + tag + "::" + flowTag + "]]  " + message, exception);
		}
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	public static void d(@NonNull String tag, @NonNull String message) {
		if (DEBUG) {
			logger.d(TAG, "[[" + tag + "]]  " + message);
		}
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	public static void d(@NonNull String tag, @NonNull String flowTag, @NonNull String message) {
		if (DEBUG) {
			logger.d(TAG, "[[" + tag + "::" + flowTag + "]]  " + message);
		}
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void d(@NonNull String tag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.d(TAG, "[[" + tag + "]]  " + message, exception);
		}
	}

	/**
	 * Logs a message using the main developer-facing log-level (DEBUG).
	 * Good for useful debugging information.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void d(@NonNull String tag, @NonNull String flowTag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.d(TAG, "[[" + tag + "::" + flowTag + "]]  " + message, exception);
		}
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	public static void i(@NonNull String tag, @NonNull String message) {
		if (DEBUG) {
			logger.i(TAG, "[[" + tag + "]]  " + message);
		}
		addToLoggos(tag, message, INFO);
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	public static void i(@NonNull String tag, @NonNull String flowTag, @NonNull String message) {
		if (DEBUG) {
			logger.i(TAG, "[[" + tag + "::" + flowTag + "]]  " + message);
		}
		addToLoggos(tag + "::" + flowTag, message, INFO);
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void i(@NonNull String tag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.i(TAG, "[[" + tag + "]]  " + message, exception);
		}
		addToLoggos(tag, message + " ** " + getStackTrace(exception), INFO);
	}

	/**
	 * Logs a message using the lowest customer-facing log-level (INFO).
	 * This information should be understandable by businesspeople.
	 * Good for noting the successful completion of business logic.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void i(@NonNull String tag, @NonNull String flowTag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.i(TAG, "[[" + tag + "::" + flowTag + "]]  " + message, exception);
		}
		addToLoggos(tag + "::" + flowTag, message + " ** " + getStackTrace(exception), INFO);
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	public static void w(@NonNull String tag, @NonNull String message) {
		if (DEBUG) {
			logger.w(TAG, "[[" + tag + "]]  " + message);
		}
		addToLoggos(tag, message, WARN);
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	public static void w(@NonNull String tag, @NonNull String flowTag, @NonNull String message) {
		if (DEBUG) {
			logger.w(TAG, "[[" + tag + "::" + flowTag + "]]  " + message);
		}
		addToLoggos(tag + "::" + flowTag, message, WARN);
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void w(@NonNull String tag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.w(TAG, "[[" + tag + "]]  " + message, exception);
		}
		addToLoggos(tag, message + " ** " + getStackTrace(exception), WARN);
	}

	/**
	 * Logs a message using the middle customer-facing log-level (WARNING).
	 * This information should be understandable by businesspeople.
	 * Good for noting the **successful** handling of a failure case.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void w(@NonNull String tag, @NonNull String flowTag, @NonNull String message, @NonNull Throwable exception) {
		if (DEBUG) {
			logger.w(TAG, "[[" + tag + "::" + flowTag + "]]  " + message, exception);
		}
		addToLoggos(tag + "::" + flowTag, message + " ** " + getStackTrace(exception), WARN);
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 */
	public static void e(@NonNull String tag, @NonNull String message) {
		logger.e(TAG,"[[" + tag + "]]  " + message);
		sendToLoggosNow(tag, message, ERROR);
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 */
	public static void e(@NonNull String tag, @NonNull String flowTag, @NonNull String message) {
		logger.e(TAG,"[[" + tag + "::" + flowTag + "]]  " + message);
		sendToLoggosNow(tag + "::" + flowTag, message, ERROR);
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void e(@NonNull String tag, @NonNull String message, @NonNull Throwable exception) {
		logger.e(TAG,"[[" + tag + "]]  " + message, exception);
		sendToLoggosNow(tag, message + " ** " + getStackTrace(exception), ERROR);
	}

	/**
	 * Logs a message using the highest customer-facing log-level (ERROR).
	 * This information should be understandable by businesspeople.
	 * Good for noting failures in logic that have interrupted some piece of functionality.
	 * @param tag A tag identifying the class that's logging this message
	 * @param flowTag A tag identifying what that class was doing when it logged this message
	 * @param message A message to write to system logs
	 * @param exception A relevant Exception to write to the logs with a full stacktrace
	 */
	public static void e(@NonNull String tag, @NonNull String flowTag, @NonNull String message, @NonNull Throwable exception) {
		logger.e(TAG,"[[" + tag + "::" + flowTag + "]]  " + message, exception);
		sendToLoggosNow(tag + "::" + flowTag, message + " ** " + getStackTrace(exception), ERROR);
	}

	private static String getStackTrace(Throwable aThrowable) {
		if (aThrowable == null) return "";
		final Writer result = new StringWriter();
		final PrintWriter printWriter = new PrintWriter(result);
		aThrowable.printStackTrace(printWriter);
		return result.toString();
	}

    /**
     * Send feature usage statistics to Loggos which will eventually picked by Kibana
     * @param classTAG Class from where we are sending message
     * @param message message to be sent
     */
    public static void sendUsageStatistics(String classTAG, String message) {
        if (isLoggosEnabled) {
            LoggosMessage loggosMessage = new LoggosMessage(INFO, LOGGOS_CONTEXT, message, classTAG);
            loggosMessage.setSdkVersion(BuildConfig.VERSION_NAME);
            loggosMessage.setIsFeatureStatistics(true);
            Loggos loggos = Infra.instance.getLoggos();
            if (loggos != null) {
                loggos.sendMessageImmediately(loggosMessage);
            }
        }
    }

    private static void addToLoggos(String classTAG, String message, String level) {
        if (isLoggosEnabled) {
            LoggosMessage loggosMessage = new LoggosMessage(level, LOGGOS_CONTEXT, message, classTAG);
            Loggos loggos = Infra.instance.getLoggos();
            if (loggos != null) {
                loggos.addMessage(loggosMessage);
            }
        }
	}

	private static void sendToLoggosNow(String classTAG, String message, String level) {
		if(isLoggosEnabled) {
			LoggosMessage loggosMessage = new LoggosMessage(level, LOGGOS_CONTEXT, message, classTAG);
			Loggos loggos = Infra.instance.getLoggos();
			if (loggos != null) {
				loggos.sendMessageImmediately(loggosMessage);
			}
		}
	}

	public class FlowTags {
		public static final String DIALOGS = "MULTI_DIALOG_FLOW";
		public static final String LOGIN = "LOGIN_FLOW";
		public static final String DECRYPTION = "DECRYPTION";
		public static final String QUICK_REPLIES = "QuickReplies";
	}
}
