package com.liveperson.infra.utils;

import android.content.Context;
import androidx.annotation.Nullable;

import com.liveperson.infra.log.LPMobileLog;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

public class FileUtils {
    private static final String TAG = "FileUtils";

    private static String IMAGES_FOLDER = "images/";
    private static String VOICE_FOLDER = "voice/";
    private static String DOCUMENTS_FOLDER = "documents/";

    public static boolean copyFile(File privateDirectory, File publicDirectory) {
        FileInputStream in = null;
        FileOutputStream out = null;
        try {
            in = new FileInputStream(privateDirectory);
            out = new FileOutputStream(publicDirectory);
            byte[] b = new byte[1024 * 5];
            int len = 0;
            while ((len = in.read(b)) > 0) {
                out.write(b, 0, len);
            }
        } catch (Exception e) {
            LPMobileLog.e(TAG, "Exception copying file.", e);
        } finally {
            try {
                if (out != null) {
                    out.close();
                }
            } catch (IOException e) {
                LPMobileLog.e(TAG, "Exception closing FileOutputStream", e);
            }
            try {
                if (in != null) {
                    in.close();
                }
            } catch (IOException e) {
                LPMobileLog.e(TAG, "Exception closing FileOutputStream", e);
            }
        }
        return privateDirectory.getName().equals(publicDirectory.getName());
    }

    /**
     * Create new directory inside app's private directory and file if does not exists
     * @param context application context
     * @param path location of file to be created
     * @param filename Name of file to be created
     * @return
     */
    @Nullable
    public static File getFilePath(Context context, String path, String filename) {
        // Define directory path and file path
        File privateDir = new File(context.getApplicationContext().getFilesDir() + path);
        File filePath = new File(privateDir, filename);

        // Create directory for the current brandId
        if (!privateDir.exists()) {
            if (!privateDir.mkdirs()) {
                LPMobileLog.e(TAG, "getFilePath: folder could not be created");
                return null;
            }
        }
        return filePath;
    }


    /**
     * Delete all downloaded files shared with agent
     */
    public static void deleteAllSharedFiles(Context context) {

        LPMobileLog.d(TAG,"deleting all shared files with agent");
        deleteFolder(context, IMAGES_FOLDER);
        deleteFolder(context, VOICE_FOLDER);
        deleteFolder(context, DOCUMENTS_FOLDER);
    }

    /**
     * Delete all files stored on the device. This should be called when the user logs out his account
     *
     * @param folderName
     */
    private static void deleteFolder(Context context, String folderName) {

        File dirToDelete = new File(context.getFilesDir() + "/" + folderName + "/");

        LPMobileLog.d(TAG, "delete" + folderName + "Folder: Trying to delete the" + folderName + "folder:" + dirToDelete.getAbsolutePath());

        deleteRecursively(dirToDelete);
    }

    /**
     * Recursively delete all files (Images and documents) and folders from the device.
     *
     * @param fileOrDirectory
     */
    private static void deleteRecursively(File fileOrDirectory) {

        if (fileOrDirectory.isDirectory()) {
            LPMobileLog.d(TAG, "deleteRecursive: deleting directory: " + fileOrDirectory.getAbsolutePath());

            for (File child : fileOrDirectory.listFiles()) {
                LPMobileLog.d(TAG, "deleteRecursive: deleting file: " + child.getAbsolutePath());
                deleteRecursively(child);
            }
        }

        //noinspection ResultOfMethodCallIgnored
        fileOrDirectory.delete();
    }
}
