package com.liveperson.infra.network;



/**
 * Created by ofira on 5/8/16.
 */
public class ExponentialBackOff implements BackOff {

    /**
     * The default initial interval value in milliseconds (0.5 seconds).
     */
    private static final int DEFAULT_INITIAL_INTERVAL_MILLIS = 500;

    /**
     * The default randomization factor 0
     */
    private static final double DEFAULT_RANDOMIZATION_FACTOR = 0;

    /**
     * The default multiplier value (1.5 which is 50% increase per back off).
     */
    private static final double DEFAULT_MULTIPLIER = 1.5;

    /**
     * The default maximum back off time in milliseconds (1 minute).
     */
    private static final int DEFAULT_MAX_INTERVAL_MILLIS = 60000;

    /**
     * The default maximum elapsed time in milliseconds (15 minutes).
     */
    private static final int DEFAULT_MAX_ELAPSED_TIME_MILLIS = 900000;


    /**
     * The initial retry interval in milliseconds
     */
    private final int mInitialIntervalMillis;


    /**
     * The current retry interval in milliseconds.
     */
    private int mCurrentIntervalMillis;


    /**
     * The randomization factor to use for creating a range around the retry interval.
     * <p/>
     * <p>
     * A randomization factor of 0.5 results in a random period ranging between 50% below and 50%
     * above the retry interval.
     * </p>
     */
    private final double mRandomizationFactor;

    /**
     * The value to multiply the current interval with for each retry attempt.
     */
    private final double mMultiplier;

    /**
     * The maximum value of the back off period in milliseconds. Once the retry interval reaches this
     * value it stops increasing.
     */
    private final int mMaxIntervalMillis;

    /**
     * The maximum elapsed time after instantiating {@link ExponentialBackOff} or calling
     * {@link #reset()} after which {@link #getNextBackOffMillis()} returns {@link BackOff#STOP}.
     */
    private final int mMaxElapsedTimeMillis;



    /**
     * The system time in nanoseconds. It is calculated when an ExponentialBackOffPolicy instance is
     * created and is reset when {@link #reset()} is called.
     */
    private long mStartTimeNanos;

    private int mRetryNumber;

    private long mNextDelay;


    private ExponentialBackOff(Builder builder) {
        mInitialIntervalMillis = builder.initialIntervalMillis;
        mRandomizationFactor = builder.randomizationFactor;
        mMultiplier = builder.multiplier;
        mMaxIntervalMillis = builder.maxIntervalMillis;
        mMaxElapsedTimeMillis = builder.maxElapsedTimeMillis;
        mRetryNumber = 0;
        reset();
    }


    @Override
    public void reset() {
        mCurrentIntervalMillis = mInitialIntervalMillis;
        mStartTimeNanos = System.nanoTime();
        mRetryNumber = 0;
        mNextDelay = 0;
    }

    @Override
    public long getNextBackOffMillis() {
        return mNextDelay;
    }

    @Override
    public void calculateNextBackOffMillis() {
        // Make sure we have not gone over the maximum elapsed time.
        if (getElapsedTimeMillis() > mMaxElapsedTimeMillis) {
            mRetryNumber = 0;
            mNextDelay = STOP;
            return;
        }

        //Make sure we have not gone over the maximum delay time
        mNextDelay = getRandomValueFromInterval(mRandomizationFactor, Math.random(), mCurrentIntervalMillis);
        if(mNextDelay >= mMaxIntervalMillis){
            mRetryNumber = 0;
            mNextDelay = STOP;
            return;
        }
        mRetryNumber++;
        incrementCurrentInterval();
    }

    @Override
    public int getRetryNumber() {
        return mRetryNumber;
    }

    /**
     * Returns the elapsed time in milliseconds since an {@link ExponentialBackOff} instance is
     * created and is reset when {@link #reset()} is called.
     *
     * <p>
     * The elapsed time is computed using {@link System#nanoTime()}.
     * </p>
     */
    private long getElapsedTimeMillis() {
        return (System.nanoTime() - mStartTimeNanos) / 1000000;
    }


    /**
     * Returns a random value from the interval [randomizationFactor * currentInterval, randomizationFactor * currentInterval].
     */
    private long getRandomValueFromInterval(double randomizationFactor, double random, int currentIntervalMillis) {
        double delta = randomizationFactor * currentIntervalMillis;
        double minInterval = currentIntervalMillis - delta;
        double maxInterval = currentIntervalMillis + delta;

        // Get a random value from the range [minInterval, maxInterval].
        // The formula used below has a +1 because if the minInterval is 1 and the maxInterval is 3 then
        // we want a 33% chance for selecting either 1, 2 or 3.

        int randomValue = (int) (minInterval + (random * (maxInterval - minInterval + 1)));
        return randomValue*1000;
    }


    /**
     * Increments the current interval by multiplying it with the multiplier.
     */
    private void incrementCurrentInterval() {
        // Check for overflow, if overflow is detected set the current interval to the max interval.
        if (mCurrentIntervalMillis >= mMaxIntervalMillis / mMultiplier) {
            mCurrentIntervalMillis = mMaxIntervalMillis;
        } else {
            mCurrentIntervalMillis *= mMultiplier;
        }
    }


    public static class Builder {
        int initialIntervalMillis = DEFAULT_INITIAL_INTERVAL_MILLIS;
        double randomizationFactor = DEFAULT_RANDOMIZATION_FACTOR;
        double multiplier = DEFAULT_MULTIPLIER;
        int maxIntervalMillis = DEFAULT_MAX_INTERVAL_MILLIS;
        int maxElapsedTimeMillis = DEFAULT_MAX_ELAPSED_TIME_MILLIS;

        public Builder() {
        }

        /**
         * Builds a new instance of {@link ExponentialBackOff}.
         */
        public ExponentialBackOff build() {
            return new ExponentialBackOff(this);
        }

        /**
         * Sets the initial retry interval in milliseconds. The default value is
         * {@link #DEFAULT_INITIAL_INTERVAL_MILLIS}. Must be {@code > 0}.
         * <p/>
         * <p>
         * Overriding is only supported for the purpose of calling the super implementation and changing
         * the return type, but nothing else.
         * </p>
         */
        public Builder setInitialIntervalMillis(int initialIntervalMillis) {
            this.initialIntervalMillis = initialIntervalMillis;
            return this;
        }

        /**
         * Sets the value to multiply the current interval with for each retry attempt. The default
         * value is {@link #DEFAULT_MULTIPLIER}. Must be {@code >= 1}.
         * <p/>
         * <p>
         * Overriding is only supported for the purpose of calling the super implementation and changing
         * the return type, but nothing else.
         * </p>
         */
        public Builder setMultiplier(double multiplier) {
            this.multiplier = multiplier;
            return this;
        }

        /**
         * Sets the maximum value of the back off period in milliseconds. Once the current interval
         * reaches this value it stops increasing. The default value is
         * {@link #DEFAULT_MAX_INTERVAL_MILLIS}.
         * <p/>
         * <p>
         * Overriding is only supported for the purpose of calling the super implementation and changing
         * the return type, but nothing else.
         * </p>
         */
        public Builder setMaxIntervalMillis(int maxIntervalMillis) {
            this.maxIntervalMillis = maxIntervalMillis;
            return this;
        }

        /**
         * Sets the maximum elapsed time in milliseconds. The default value is
         * {@link #DEFAULT_MAX_ELAPSED_TIME_MILLIS}. Must be {@code > 0}.
         * <p/>
         * <p>
         * If the time elapsed since an {@link ExponentialBackOff} instance is created goes past the
         * max_elapsed_time then the method {@link #getNextBackOffMillis()} starts returning
         * {@link BackOff#STOP}. The elapsed time can be reset by calling {@link #reset()}.
         * </p>
         * <p/>
         * <p>
         * Overriding is only supported for the purpose of calling the super implementation and changing
         * the return type, but nothing else.
         * </p>
         */
        public Builder setMaxElapsedTimeMillis(int maxElapsedTimeMillis) {
            this.maxElapsedTimeMillis = maxElapsedTimeMillis;
            return this;
        }
    }
}