package com.liveperson.infra.managers

import android.content.Context
import com.liveperson.infra.auth.LPAuthenticationParams
import com.liveperson.infra.auth.LPAuthenticationType
import com.liveperson.infra.callbacks.AuthCallBack
import com.liveperson.infra.callbacks.AuthStateSubscription
import com.liveperson.infra.log.LPLog
import com.liveperson.infra.managers.ConsumerManager.AuthState.*
import com.liveperson.infra.model.Consumer
import com.liveperson.infra.model.errors.AuthError
import com.liveperson.infra.model.types.AuthFailureReason
import com.liveperson.infra.network.http.requests.AuthRequest
import com.liveperson.infra.preferences.AuthPreferences
import java.lang.Exception
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.atomic.AtomicBoolean


/**
 * Manager responsible for logging Consumers in and out of LivePerson's back-end. Takes in user detail
 * payloads in the form of LPAuthenticationParams and returns user details in the form of a Consumer
 * object.
 *
 * At some point in the future, this class should be modified to not clear all user data, but hot-swap
 * between data sets when it sees a brandId that is different from the brandId it was just working with.
 */
open class ConsumerManager(private val appContext: Context, private val brandId: String) {

	companion object {
		private const val TAG = "ConsumerManager"

		/**
		 * Extension function for nullable [Consumer] instance to receive its
		 * jwt, to prevent multiple null check.
		 *
		 * @return JWT as [String] if consumer is setup as implicit account, null otherwise.
		 */
		@JvmStatic
		fun Consumer?.getConsumerJWT(): String? {
			return this
				?.lpAuthenticationParams
				?.takeIf { it.authType == LPAuthenticationType.AUTH }
				?.hostAppJWT
		}
	}

	private var authState = NOT_AUTHENTICATED

	private val authPrefs = AuthPreferences.getInstance(appContext)

	private var activeConsumer: Consumer? = null
	private var mostRecentError: AuthError? = null

	private var activeRequest: AuthRequest? = null

	private val callbacks: MutableSet<AuthCallBack> = Collections.newSetFromMap(ConcurrentHashMap())
	private val subscriptions: MutableSet<AuthStateSubscription> = Collections.newSetFromMap(ConcurrentHashMap())
	private val isTokenExpirationHandled = AtomicBoolean(false)

	init {
		if (brandId.isNotEmpty()) {
			activeConsumer = authPrefs.getCachedConsumer(brandId)
			/* Update activeConsumer but -not- authState; we don't know for certain that we're
			 * going to be using this same user post-auth, so for now it's a cached "dirty" copy.
			 */
		}
	}

	/**
	 * Returns *true* if there is valid Consumer data; false otherwise. THIS DOES NOT INDICATE AN
	 * AUTHENTICATED USER. Check the AuthState to ensure it is also AUTHENTICATED. If the current
	 * AuthState is NOT_AUTHENTICATED, this method may still return *true* for a cached consumer.
	 * May block while waiting on a lock.
	 */
	open fun hasActiveConsumer(): Boolean {
		synchronized(this) {
			return activeConsumer != null
		}
	}

	/**
	 * Returns the most recent active Consumer. THIS DOES NOT INDICATE AN
	 * AUTHENTICATED USER. Check the AuthState to ensure it is also AUTHENTICATED. If the current
	 * AuthState is NOT_AUTHENTICATED, this method may still return a cached consumer.
	 * May block while waiting on a lock.
	 */
	open fun getActiveConsumer(): Consumer? {
		synchronized(this) {
			return activeConsumer
		}
	}

	/**
	 * Returns *true* if there is a valid and properly-authenticated Consumer; false otherwise.
	 * May block while waiting on a lock.
	 */
	fun isAuthenticated(): Boolean {
		synchronized(this) {
			return authState == AUTHENTICATED && activeConsumer != null
		}
	}

	/**
	 * Set authentication state to be NOT AUTHENTICATED when moving to background
	 */
	fun resetAuthState() {
		synchronized(this) {
			authState = NOT_AUTHENTICATED
		}
	}

	/**
	 * Set authentication state to be AUTHENTICATED from IDPTask class strictly used for SignUp flow
	 */
	fun setIsAuthenticated() {
		synchronized(this) {
			authState = AUTHENTICATED
		}
	}

	/**
	 * Internal handler function for successful authentication attempts that aren't consumer-swaps.
	 */
	private fun handleAuthSuccessful(newConsumer: Consumer) {
		LPLog.i(TAG, "Successfully logged in ${LPLog.mask(newConsumer)}")
		isTokenExpirationHandled.set(false)
		synchronized(this) {
			val oldState = authState
			val oldConsumer = activeConsumer // save before overwriting
			authState = AUTHENTICATED
			activeConsumer = newConsumer // overwrite before broadcasting, so that state matches callbacks
			mostRecentError = null

			authPrefs.setCachedConsumer(brandId, newConsumer)

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(oldState, authState, oldConsumer, newConsumer)
			}

			for (callback in callbacks) {
				callback.onAuthSuccess(newConsumer)
			}
			callbacks.clear()
		}
	}

	/**
	 * Internal handler function for consumer-swap events.
	 */
	private fun handleConsumerChange(newConsumer: Consumer) {
		LPLog.i(TAG, "Encountered unexpected consumerId; performing consumer swap from " +
				"${LPLog.mask(activeConsumer)} to ${LPLog.mask(newConsumer)}.")
		isTokenExpirationHandled.set(false)
		synchronized(this) {
			val oldConsumer = activeConsumer // save before overwriting
			authState = AUTHENTICATED
			activeConsumer = newConsumer
			mostRecentError = null

			authPrefs.setCachedConsumer(brandId, newConsumer)

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(AUTHENTICATED, AUTHENTICATED, oldConsumer, newConsumer)
			}

			for (callback in callbacks) {
				callback.onConsumerSwitch(oldConsumer!!, newConsumer)
			}
			callbacks.clear()
		}
	}

	/**
	 * Internal handler function for failed authentication attempts.
	 */
	private fun handleAuthFailure(authError: AuthError, initialAuthParams: LPAuthenticationParams?) {
		LPLog.w(TAG, "Auth failed. Reason: $authError")
		isTokenExpirationHandled.set(false)
		synchronized(this) {
			val oldState = authState
			val oldConsumer = activeConsumer // save before overwriting
			val shouldSave = initialAuthParams
				?.let { it.authType == LPAuthenticationType.AUTH && !it.authKey.isNullOrBlank() }
				?.takeIf { authError.statusCode == 401 }
				?: false

			authState = AUTH_FAILED
			activeConsumer = oldConsumer?.takeIf { shouldSave } // overwrite before broadcasting, so that state matches callbacks
			mostRecentError = authError

			authPrefs.setCachedConsumer(brandId, null)

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(oldState, authState, oldConsumer, null)
			}

			for (callback in callbacks) {
				callback.onAuthFailed(authError)
			}
			callbacks.clear()
		}
	}

	/**
	 * Method used to handle jwt-token expiration, ones implicit user's jwt is expired.
	 *
	 * @param exception - error thrown because of token expiration. Token could be expired while
	 * history retrieval, file uploading/downloading, etc.
	 * @param block     - executable code/callback required to start token refreshing. Note:
	 * if token started execution block would be ignored to prevent multiple startups of token
	 * renewal process.
	 */
	fun handleTokenExpiration(exception: Exception, block: () -> Unit) {

		if (isTokenExpirationHandled.compareAndSet(false, true)) {
			block()
		}
		synchronized(this) {
			val oldState = authState
			val oldConsumer = activeConsumer // save before overwriting

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(oldState, EXPIRED, oldConsumer, null)
			}

			for (callback in callbacks) {
				callback.onAuthFailed(AuthError(AuthFailureReason.TOKEN_EXPIRED, exception, 401))
			}
			callbacks.clear()
		}
	}

	/**
	 * Logs in a Consumer, either a new one or old one. If the authentication attempt is successful,
	 * the Consumer's details will be encrypted and cached for future reference. The optional callback
	 * parameter functions identically to registering a callback separately and then calling login.
	 * May block while waiting on a lock.
	 */
	fun login(authParams: LPAuthenticationParams?, idpDomain: String?, hostVersion: String?, certs: List<String?>?,
			  unAuthConnectorId: String?, authConnectorId: String?, performSteUp: Boolean, callback: AuthCallBack?) {

		synchronized(this) {
			if (activeRequest != null) {
				activeRequest!!.cancelAuth() // if we're given new auth details, always use the latest
				activeRequest = null
			}

			val oldState = authState
			authState = AUTH_IN_PROGRESS

			if (callback != null) {
				callbacks.add(callback) // we already have a system to handle these
			}

			activeRequest = AuthRequest(appContext, brandId, idpDomain, authParams,
					hostVersion, certs, unAuthConnectorId, authConnectorId, performSteUp,

					object : AuthCallBack {

						override fun onAuthSuccess(consumer: Consumer) {
							synchronized(this) {
								if (activeConsumer != null && !activeConsumer!!.consumerId.isNullOrEmpty()
										&& activeConsumer != consumer) {
									handleConsumerChange(consumer)
								} else {
									handleAuthSuccessful(consumer)
								}
							}
						}

						override fun onConsumerSwitch(oldConsumer: Consumer, newConsumer: Consumer) {
							// We handle consumer switch in IdpTask
						}

						override fun onAuthFailed(error: AuthError) {
							handleAuthFailure(error, authParams)
						}
					})

			activeRequest!!.authenticate()

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(oldState, authState, activeConsumer, activeConsumer)
			}
		}
	}

	/**
	 * Subscribes to all future changes to Auth State and active Consumer(s) until further notice.
	 * Current state will not be returned, but all future updates will.
	 *
	 * A state change from AUTHENTICATED to AUTHENTICATED with two different consumers indicates a
	 * USER_CHANGE event, and should be handled carefully.
	 *
	 * May block while waiting on a lock.
	 */
	open fun subscribeToAuthStateChanges(subscription: AuthStateSubscription) {
		synchronized(this) {
			subscriptions.add(subscription)
		}
	}

	/**
	 * Removes a previously-registered Subscription.
	 * May block while waiting on a lock.
	 */
	fun unsubscribeFromAuthStateChanges(subscription: AuthStateSubscription): Boolean {
		synchronized(this) {
			return subscriptions.remove(subscription)
		}
	}

	/**
	 * Cleans up the current instance and destroys its resources, ensuring no callbacks can be fired
	 * erroneously, and makes one final notice to the Subscriptions list that the consumer has
	 * become null.
	 * May block while waiting on a lock.
	 */
	fun shutdown() {
		synchronized(this) {
			if (activeRequest != null) {
				activeRequest!!.cancelAuth()
			}
			activeRequest = null

			val oldState = authState
			val oldConsumer = activeConsumer // save before overwriting

			authState = NOT_AUTHENTICATED
			activeConsumer = null

			for (subscription in subscriptions) {
				subscription.onAuthStateChanged(oldState, authState, oldConsumer, null)
			}

			callbacks.clear()
			subscriptions.clear()
		}
	}

	fun getCurrentConsumerAuthType(brandId: String): LPAuthenticationType? {
		return authPrefs.getCurrentAuthType(brandId)
	}

	/**
	 * Clears all stored Consumer and Authentication data from persistent storage, and then calls
	 * [ConsumerManager.shutdown]. May block while waiting on a lock.
	 */
	open fun clearDataAndShutdown() {
		synchronized(this) {
			authPrefs.clearAll()
			shutdown()
		}
	}

	enum class AuthState {
		NOT_AUTHENTICATED, AUTH_IN_PROGRESS, AUTHENTICATED, AUTH_FAILED, EXPIRED
	}
}