package com.liveperson.infra.preferences

import android.content.Context
import android.content.SharedPreferences
import com.liveperson.infra.log.LPLog
import java.util.*

/**
 * Parent class to cache all SDK logs. You should inherit this class when need to cache specific type
 * of logs (Loggos, EventManager, etc). Currently used to store Event Manager logs
 */
open class LogPreferences {

    companion object {
        private const val TAG = "LogPreferences"
        const val LP_LOG_SHARED_PREF_FILENAME = "lp_log_shared_pref"
    }

    private var preferences: SharedPreferences? = null

    /**
     * Initialize LP Log shared preferences.
     */
    fun initialize(applicationContext: Context) {
        if (preferences == null) {
            LPLog.i(TAG, "Initializing log preferences....")
            preferences = applicationContext.getSharedPreferences(
                    LP_LOG_SHARED_PREF_FILENAME, Context.MODE_PRIVATE)
        }
    }

    /**
     * Store logs in shared preferences for a given key
     */
    fun storeLogsInPreferences(key: String, logs: HashSet<String>) {
        LPLog.d(TAG, "Storing logs in preferences")
        preferences?.edit()?.putStringSet(key, logs)?.apply()
    }

    /**
     * Return logs from shared preferences for a given key
     */
    fun getLogsFromPreferences(key: String): Set<String>? {
        LPLog.d(TAG, "Return logs from preferences")
        return preferences?.getStringSet(key, null)
    }


    /**
     * Delete logs from shared preferences for a given key
     */
    fun removeLogsFromPreferences(key: String) {
        LPLog.d(TAG, "Clear logs from preferences for a key: {$key}")
        preferences?.edit()?.remove(key)?.apply()
    }

    /**
     * Generate a unique key for log events per service type (Loggos or EventManager)
     */
    fun buildKey(brandId: String, logServiceType: String): String {
        return "$logServiceType::$brandId"
    }

    /**
     * Clear entire log shared preference
     */
    fun clearAll() {
        LPLog.d(TAG, "Clear all logs from preferences")
        preferences?.edit()?.clear()?.apply()
    }
}